package main

import (
	"consistent_sql/protocol"
	"consistent_sql/testutils"
	"encoding/binary"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"os"
	"path/filepath"
	"strings"
	"testing"
	"time"
)

// 添加保存数据包的函数
func saveDataPacket(data []byte, messageType string) error {
	// 创建日志目录
	logDir := "mongodb_data_packets"
	if err := os.MkdirAll(logDir, 0755); err != nil {
		return fmt.Errorf("创建日志目录失败: %v", err)
	}

	// 生成文件名，包含时间戳和类型
	timestamp := time.Now().Format("20060102_150405.000")
	filename := filepath.Join(logDir, fmt.Sprintf("mongo_packet_%s_%s.bin", messageType, timestamp))

	// 保存二进制数据
	if err := ioutil.WriteFile(filename, data, 0644); err != nil {
		return fmt.Errorf("保存数据包失败: %v", err)
	}

	// 保存十六进制视图
	hexFilename := filepath.Join(logDir, fmt.Sprintf("mongo_packet_%s_%s.hex", messageType, timestamp))
	hexView := testutils.FormatPacketHex(data)
	if err := ioutil.WriteFile(hexFilename, []byte(hexView), 0644); err != nil {
		return fmt.Errorf("保存十六进制视图失败: %v", err)
	}

	fmt.Printf("数据包已保存: %s (二进制) 和 %s (十六进制视图)\n", filename, hexFilename)
	return nil
}

// 添加解析并打印数据包字段的函数
func analyzeMongoPacket(data []byte) string {
	if len(data) < 20 {
		return "数据包太短，无法完整解析"
	}

	var result strings.Builder
	result.WriteString("MongoDB数据包字段详细解析:\n")
	result.WriteString("=======================\n\n")

	offset := 0

	// 1. 解析消息类型
	if offset >= len(data) {
		return result.String() + "数据包不完整，无法解析消息类型"
	}
	messageType := data[offset]
	result.WriteString(fmt.Sprintf("消息类型: 1 字节 (偏移量: %d) 值: %d\n\n", offset, messageType))
	offset++

	// 2. 解析行数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析行数量"
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	result.WriteString(fmt.Sprintf("行数量: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, rowCount))
	offset += 4

	// 3. 解析GTID
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析GTID长度"
	}
	gtidLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("GTID长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, gtidLen))
	offset += 4

	if offset+gtidLen > len(data) {
		return result.String() + "数据包不完整，无法读取GTID内容"
	}
	gtid := string(data[offset : offset+gtidLen])
	result.WriteString(fmt.Sprintf("GTID内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+gtidLen-1, gtid))
	offset += gtidLen

	// 4. 解析binlog文件
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析binlog文件"
	}
	binlogFile := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("binlog文件: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, binlogFile))
	offset += 4

	// 5. 解析数据类型
	if offset >= len(data) {
		return result.String() + "数据包不完整，无法解析数据类型"
	}
	dataType := int8(data[offset])
	var typeDesc string
	switch dataType {
	case 13:
		typeDesc = "插入"
	case 14:
		typeDesc = "更新"
	case 15:
		typeDesc = "删除"
	default:
		typeDesc = "未知"
	}
	result.WriteString(fmt.Sprintf("数据类型: 1 字节 (偏移量: %d) 值: %d (%s)\n\n", offset, dataType, typeDesc))
	offset++

	// 6. 解析数据库名
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析数据库名长度"
	}
	dbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("数据库名长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, dbNameLen))
	offset += 4

	if offset+dbNameLen > len(data) {
		return result.String() + "数据包不完整，无法读取数据库名内容"
	}
	dbName := string(data[offset : offset+dbNameLen])
	result.WriteString(fmt.Sprintf("数据库名内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+dbNameLen-1, dbName))
	offset += dbNameLen

	// 7. 解析表名
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析表名长度"
	}
	tbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("表名长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, tbNameLen))
	offset += 4

	if offset+tbNameLen > len(data) {
		return result.String() + "数据包不完整，无法读取表名内容"
	}
	tbName := string(data[offset : offset+tbNameLen])
	result.WriteString(fmt.Sprintf("表名内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+tbNameLen-1, tbName))
	offset += tbNameLen

	// 8. 解析GTID（再次写入）
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析第二个GTID长度"
	}
	gtidLen2 := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("GTID长度(再次): %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, gtidLen2))
	offset += 4

	if offset+gtidLen2 > len(data) {
		return result.String() + "数据包不完整，无法读取第二个GTID内容"
	}
	gtid2 := string(data[offset : offset+gtidLen2])
	result.WriteString(fmt.Sprintf("GTID内容(再次): (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+gtidLen2-1, gtid2))
	offset += gtidLen2

	// 9. 解析SQL（MongoDB没有SQL，值为0）
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析SQL值"
	}
	sqlValue := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("SQL值: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, sqlValue))
	offset += 4

	// 10. 解析列数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析列数量"
	}
	columnNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("列数量: %d 字节 (偏移量: %d-%d) 值: %d\n\n", 4, offset, offset+3, columnNum))
	offset += 4

	// 11. 解析列名
	columnNames := make([]string, 0, columnNum)
	result.WriteString("列名信息:\n")
	for i := 0; i < columnNum; i++ {
		if offset+4 > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d个列名长度", i+1)
		}
		colNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
		result.WriteString(fmt.Sprintf("  第%d列名长度: %d 字节 (偏移量: %d-%d) 值: %d\n", i+1, 4, offset, offset+3, colNameLen))
		offset += 4

		if offset+colNameLen > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法读取第%d个列名内容", i+1)
		}
		colName := string(data[offset : offset+colNameLen])
		columnNames = append(columnNames, colName)
		result.WriteString(fmt.Sprintf("  第%d列名内容: (偏移量: %d-%d) 值: '%s'\n", i+1, offset, offset+colNameLen-1, colName))
		offset += colNameLen
	}
	result.WriteString("\n")

	// 12. 解析列属性
	result.WriteString("列属性信息:\n")
	for i := 0; i < columnNum; i++ {
		if offset+4 > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d个列属性", i+1)
		}
		colAttr := binary.LittleEndian.Uint32(data[offset : offset+4])
		result.WriteString(fmt.Sprintf("  第%d列属性: %d 字节 (偏移量: %d-%d) 值: %d\n", i+1, 4, offset, offset+3, colAttr))
		offset += 4
	}
	result.WriteString("\n")

	// 13. 解析行数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析具体行数量"
	}
	rowNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("具体行数量: %d 字节 (偏移量: %d-%d) 值: %d\n\n", 4, offset, offset+3, rowNum))
	offset += 4

	// 14. 解析行数据
	result.WriteString("行数据:\n")
	for i := 0; i < rowNum; i++ {
		result.WriteString(fmt.Sprintf("  第%d行:\n", i+1))
		for j := 0; j < columnNum; j++ {
			if offset+4 > len(data) {
				return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d行第%d列的长度", i+1, j+1)
			}
			valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
			result.WriteString(fmt.Sprintf("    列'%s'值长度: %d 字节 (偏移量: %d-%d) 值: %d\n",
				columnNames[j], 4, offset, offset+3, valueLen))
			offset += 4

			if offset+valueLen > len(data) {
				return result.String() + fmt.Sprintf("数据包不完整，无法读取第%d行第%d列的内容", i+1, j+1)
			}
			value := string(data[offset : offset+valueLen])
			result.WriteString(fmt.Sprintf("    列'%s'值内容: (偏移量: %d-%d) 值: '%s'\n",
				columnNames[j], offset, offset+valueLen-1, value))
			offset += valueLen
		}
		result.WriteString("\n")
	}

	// 15. 解析旧值（对于UPDATE操作）
	if dataType == 14 && offset < len(data) {
		result.WriteString("旧值数据 (UPDATE操作):\n")
		for i := 0; i < rowNum; i++ {
			result.WriteString(fmt.Sprintf("  第%d行旧值:\n", i+1))
			for j := 0; j < columnNum; j++ {
				if offset+4 > len(data) {
					return result.String() + fmt.Sprintf("数据包不完整，无法解析旧值第%d行第%d列的长度", i+1, j+1)
				}
				valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
				result.WriteString(fmt.Sprintf("    列'%s'旧值长度: %d 字节 (偏移量: %d-%d) 值: %d\n",
					columnNames[j], 4, offset, offset+3, valueLen))
				offset += 4

				if offset+valueLen > len(data) {
					return result.String() + fmt.Sprintf("数据包不完整，无法读取旧值第%d行第%d列的内容", i+1, j+1)
				}
				value := string(data[offset : offset+valueLen])
				result.WriteString(fmt.Sprintf("    列'%s'旧值内容: (偏移量: %d-%d) 值: '%s'\n",
					columnNames[j], offset, offset+valueLen-1, value))
				offset += valueLen
			}
			result.WriteString("\n")
		}
	}

	// 添加剩余未解析的字节大小提示
	if offset < len(data) {
		result.WriteString(fmt.Sprintf("注意: 数据包中还有 %d 字节未解析\n", len(data)-offset))
	}

	return result.String()
}

func TestMongoStreamFormat(t *testing.T) {
	// 定义并解析命令行参数
	serverUnixFile := flag.String("serverUnixFile", "/Users/helay/Documents/GitHub/Kunlun-XPanel/KunlunMonitor/go/consistent_sql/socket.sock", "Unix socket 文件路径")
	workType := flag.String("workType", "create", "任务类型：create 或 status")
	jobId := flag.String("jobId", fmt.Sprintf("mongo_test_%d", time.Now().Unix()), "任务ID")
	// mongoURL := flag.String("mongoURL", "mongodb://192.168.0.19:27018", "MongoDB URL")
	// mongoDatabase := flag.String("mongoDatabase", "test", "MongoDB 数据库名")
	// mongoCollection := flag.String("mongoCollection", "c1", "MongoDB 集合名")

	mongoURL := flag.String("mongoURL", "mongodb://192.168.0.136:27017", "MongoDB URL")
	mongoDatabase := flag.String("mongoDatabase", "testdb", "MongoDB 数据库名")
	mongoCollection := flag.String("mongoCollection", "testtb", "MongoDB 集合名")

	flag.Parse()

	fmt.Printf("连接到 Unix socket: %s\n", *serverUnixFile)
	fmt.Printf("任务ID: %s, 操作类型: %s\n", *jobId, *workType)
	fmt.Println("测试将持续运行直到被手动中断 (按 Ctrl+C 终止)")

	// 连接到 Unix socket
	conn, err := net.Dial("unix", *serverUnixFile)
	if err != nil {
		t.Fatalf("连接到Source失败: %v", err)
	}
	defer conn.Close()

	// 构建MongoDB任务请求
	mongoTask := map[string]interface{}{
		"job_id":       *jobId,
		"url":          *mongoURL,
		"db_user":      "",
		"db_passwd":    "",
		"db_database":  *mongoDatabase,
		"db_table":     *mongoCollection,
		"dump_db_type": "mongodb",
		"gtid":         "000000000000000000000000", // 使用有效的MongoDB ObjectID格式
		// "gtid":         "66bdc8b6809ad7cf6d1cf3f4", // 使用有效的MongoDB ObjectID格式
		"event_type": *workType,
	}

	// 如果是status请求，只需要提供job_id
	if *workType == "status" {
		mongoTask = map[string]interface{}{
			"job_id":       *jobId,
			"dump_db_type": "mongodb",
			"event_type":   "status",
		}
	}

	taskData, _ := json.Marshal(mongoTask)
	fmt.Printf("发送任务数据: %s\n", string(taskData))

	// 构建消息
	requestMsg := &protocol.Message{
		Type: protocol.MReqdumpjob,
		Data: string(taskData),
	}

	marshal, err := json.Marshal(requestMsg)
	if err != nil {
		t.Fatalf("序列化消息失败: %v", err)
	}

	// 发送请求
	if err := protocol.SendMsg(conn, marshal); err != nil {
		t.Fatalf("发送数据到Source失败: %v", err)
	}

	fmt.Println("已发送请求，等待响应...")

	// 循环读取响应
	messageCount := 0

	for {
		if err := readAndProcessMongoMsg(conn); err != nil {
			// 仅在首次连接出错时终止测试
			if messageCount == 0 {
				t.Fatalf("首次处理消息出错: %v", err)
			}

			// 连接断开或出错时尝试重新连接
			fmt.Printf("读取消息出错: %v，尝试重新连接...\n", err)

			// 关闭旧连接
			conn.Close()

			// 尝试重新连接
			newConn, reconnErr := net.Dial("unix", *serverUnixFile)
			if reconnErr != nil {
				fmt.Printf("重新连接失败: %v，等待5秒后重试\n", reconnErr)
				time.Sleep(5 * time.Second)
				continue
			}

			conn = newConn
			defer conn.Close()

			// 重新发送任务请求
			if err := protocol.SendMsg(conn, marshal); err != nil {
				fmt.Printf("重新发送任务请求失败: %v，等待5秒后重试\n", err)
				time.Sleep(5 * time.Second)
				continue
			}

			fmt.Println("已重新连接并发送请求，继续监听...")
		}

		messageCount++
		fmt.Printf("已成功接收 %d 条消息\n", messageCount)

		// 给系统一点时间处理，避免CPU占用过高
		time.Sleep(100 * time.Millisecond)
	}
}

// readAndProcessMongoMsg 负责读取和处理从 Unix socket 接收到的消息
func readAndProcessMongoMsg(conn net.Conn) error {
	// 设置读取超时
	conn.SetReadDeadline(time.Now().Add(10 * time.Minute)) // 增加超时时间到10分钟
	defer conn.SetReadDeadline(time.Time{})                // 重置为不超时

	// 读取消息长度的4个字节
	lengthBytes := make([]byte, 4)
	_, err := io.ReadFull(conn, lengthBytes)
	if err != nil {
		// 检查是否为网络错误
		if ne, ok := err.(net.Error); ok && ne.Timeout() {
			return fmt.Errorf("读取消息长度超时: %w", err)
		}

		// 检查连接是否已关闭
		if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
			return fmt.Errorf("连接已关闭或远程主机断开连接: %w", err)
		}

		return fmt.Errorf("读取消息长度失败: %w", err)
	}

	// 解析消息长度
	length := binary.LittleEndian.Uint32(lengthBytes)

	// 检查消息长度是否合理（增加到50MB）
	maxSize := uint32(50 * 1024 * 1024) // 50MB
	if length > maxSize {
		return fmt.Errorf("消息长度异常: %d 字节 (超过最大限制 %d 字节)", length, maxSize)
	}

	// 分配缓冲区
	data := make([]byte, length)

	// 使用 io.ReadFull 确保读取完整的数据
	_, err = io.ReadFull(conn, data)
	if err != nil {
		// 检查是否为网络错误
		if ne, ok := err.(net.Error); ok && ne.Timeout() {
			return fmt.Errorf("读取消息数据超时: %w", err)
		}

		// 检查连接是否已关闭
		if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
			return fmt.Errorf("连接已关闭或远程主机断开连接: %w", err)
		}

		return fmt.Errorf("读取消息数据失败: %w", err)
	}

	// 检查消息的第一个字节，确定消息类型
	if len(data) > 0 {
		// 检查是否为二进制协议消息
		if data[0] == byte(protocol.MReqrowmsg) {
			// 这是一个二进制数据行消息
			fmt.Printf("收到MReqrowmsg (二进制数据行消息)，长度: %d 字节\n", len(data))

			// 保存完整的数据流
			if err := saveDataPacket(data, "binary"); err != nil {
				fmt.Printf("保存数据包失败: %v\n", err)
			}

			// 创建并保存详细的字段解析
			analysis := analyzeMongoPacket(data)
			analysisFilename := filepath.Join("mongodb_data_packets",
				fmt.Sprintf("mongo_packet_analysis_%s.txt", time.Now().Format("20060102_150405.000")))
			if err := ioutil.WriteFile(analysisFilename, []byte(analysis), 0644); err != nil {
				fmt.Printf("保存字段分析失败: %v\n", err)
			} else {
				fmt.Printf("数据包字段分析已保存到: %s\n", analysisFilename)
			}

			// 在控制台显示前500个字符的分析结果
			analysisPreview := analysis
			if len(analysisPreview) > 500 {
				analysisPreview = analysisPreview[:500] + "...(更多内容见文件)"
			}
			fmt.Println("\n数据包分析预览:")
			fmt.Println(analysisPreview)

			// 确认消息采用SendResponse帮助函数确保格式一致
			if err := protocol.SendResponse(conn, protocol.MRsprowmsg, "received"); err != nil {
				fmt.Printf("发送确认消息失败: %v\n", err)
			} else {
				fmt.Println("确认消息已发送，等待下一包数据...")
			}

			return nil
		} else {
			// 如果不是MReqrowmsg，记录接收到的消息类型
			fmt.Printf("收到非MReqrowmsg消息，消息类型: %d\n", data[0])
		}
	}

	// 如果不是二进制消息，尝试解析为JSON
	fmt.Println("收到消息:", string(data))

	// 检查是否为JSON格式
	if !testutils.IsValidJSON(data) {
		fmt.Printf("收到非JSON格式数据，长度: %d 字节\n", len(data))
		// 保存数据以便后续分析
		if err := saveDataPacket(data, "unknown"); err != nil {
			fmt.Printf("保存数据包失败: %v\n", err)
		}
		return nil
	}

	// 解析JSON消息
	var message protocol.Message
	if err := json.Unmarshal(data, &message); err != nil {
		fmt.Printf("无法解析的消息内容: %s\n", string(data))
		return fmt.Errorf("解析消息失败: %w", err)
	}

	// 处理不同类型的消息
	switch message.Type {
	case protocol.MRspdumpjob:
		fmt.Println("收到任务响应:", message.Data)
		saveDataPacket([]byte(message.Data), "response")
	case protocol.MRspstatemsg:
		fmt.Println("收到状态响应:", message.Data)
		saveDataPacket([]byte(message.Data), "status")
	case protocol.MReqrowmsg:
		fmt.Println("收到MReqrowmsg JSON消息:", message.Data)

		// 确认消息采用SendResponse帮助函数确保格式一致
		if err := protocol.SendResponse(conn, protocol.MRsprowmsg, "received"); err != nil {
			fmt.Printf("发送确认消息失败: %v\n", err)
		} else {
			fmt.Println("确认消息已发送，等待下一包数据...")
		}

		saveDataPacket([]byte(message.Data), "row_message")
	default:
		fmt.Println("收到未知类型JSON消息:", message.Type)
		saveDataPacket([]byte(message.Data), "unknown_json")
	}

	return nil
}

// 尝试解析字符串流格式的数据
func parseStreamFormat(data []byte) {
	if len(data) < 10 {
		fmt.Println("数据太短，无法解析")
		return
	}

	// 首先偏移量从0开始
	offset := 0

	// 解析消息类型
	messageType := data[offset]
	fmt.Printf("消息类型: %d\n", messageType)
	offset++

	// 解析行数量
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析行数量")
		return
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	fmt.Printf("行数量: %d\n", rowCount)
	offset += 4

	// 解析GTID
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析GTID长度")
		return
	}
	gtidLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+gtidLen > len(data) {
		fmt.Println("数据不完整，无法读取GTID")
		return
	}
	gtid := string(data[offset : offset+gtidLen])
	offset += gtidLen

	// 解析binlog文件值（MongoDB为0）
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析binlog文件值")
		return
	}
	binlogFile := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4

	// 解析数据类型
	if offset >= len(data) {
		fmt.Println("数据不完整，无法解析数据类型")
		return
	}
	dataType := int8(data[offset])
	var typeStr string
	switch dataType {
	case 13:
		typeStr = "插入(13)"
	case 14:
		typeStr = "更新(14)"
	case 15:
		typeStr = "删除(15)"
	default:
		typeStr = fmt.Sprintf("未知(%d)", dataType)
	}
	offset++

	// 解析数据库名
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析数据库名长度")
		return
	}
	dbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+dbNameLen > len(data) {
		fmt.Println("数据不完整，无法读取数据库名")
		return
	}
	dbName := string(data[offset : offset+dbNameLen])
	offset += dbNameLen

	// 解析表名
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析表名长度")
		return
	}
	tbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+tbNameLen > len(data) {
		fmt.Println("数据不完整，无法读取表名")
		return
	}
	tbName := string(data[offset : offset+tbNameLen])

	fmt.Printf("解析结果 - 数据库: %s, 表: %s, GTID: %s, 操作类型: %s, Binlog: %d\n",
		dbName, tbName, gtid, typeStr, binlogFile)
}

// min 返回两个整数中的较小者
func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

// TestParseStreamFormat 测试字符串流格式解析功能
func TestParseStreamFormat(t *testing.T) {
	// 创建一个模拟的数据流
	dbName := "testdb"
	tbName := "testtable"
	gtid := "testgtid123"

	// 构建模拟数据
	data := make([]byte, 0)

	// 1. 添加消息类型
	messageType := byte(protocol.MRsprowmsg) // 使用正确的消息类型
	data = append(data, messageType)

	// 2. 添加行数
	rowCount := uint32(1)
	lenBytes := make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, rowCount)
	data = append(data, lenBytes...)

	// 3. 添加GTID长度和GTID
	gtidBytes := []byte(gtid)
	gtidLen := uint32(len(gtidBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, gtidLen)
	data = append(data, lenBytes...)
	data = append(data, gtidBytes...)

	// 4. 添加binlog文件（MongoDB为0）
	binlogFile := int32(0)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, uint32(binlogFile))
	data = append(data, lenBytes...)

	// 5. 添加数据类型
	dataType := int8(13) // 13表示插入操作
	data = append(data, byte(dataType))

	// 6. 添加数据库名长度和数据库名
	dbNameBytes := []byte(dbName)
	dbNameLen := uint32(len(dbNameBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, dbNameLen)
	data = append(data, lenBytes...)
	data = append(data, dbNameBytes...)

	// 7. 添加表名长度和表名
	tbNameBytes := []byte(tbName)
	tbNameLen := uint32(len(tbNameBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, tbNameLen)
	data = append(data, lenBytes...)
	data = append(data, tbNameBytes...)

	// 8. 再次添加GTID长度和GTID
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, gtidLen)
	data = append(data, lenBytes...)
	data = append(data, gtidBytes...)

	// 9. 添加SQL（MongoDB为0）
	sqlValue := int32(0)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, uint32(sqlValue))
	data = append(data, lenBytes...)

	// 10. 添加列数量
	columnNum := uint32(3) // MongoDB使用3列: _id, data, optype
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, columnNum)
	data = append(data, lenBytes...)

	// 11. 添加列名
	columnNames := []string{"_id", "data", "optype"}
	for _, colName := range columnNames {
		colNameBytes := []byte(colName)
		colNameLen := uint32(len(colNameBytes))
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, colNameLen)
		data = append(data, lenBytes...)
		data = append(data, colNameBytes...)
	}

	// 12. 添加列属性
	for i := 0; i < 3; i++ {
		attr := uint32(0)
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, attr)
		data = append(data, lenBytes...)
	}

	// 13. 添加行数
	rowNum := uint32(1)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, rowNum)
	data = append(data, lenBytes...)

	// 14. 添加行数据
	rowData := []string{gtid, "{\"test\":\"value\"}", "'insert'"}
	for _, value := range rowData {
		valueBytes := []byte(value)
		valueLen := uint32(len(valueBytes))
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, valueLen)
		data = append(data, lenBytes...)
		data = append(data, valueBytes...)
	}

	// 测试parseStreamFormat函数
	fmt.Println("测试parseStreamFormat函数:")
	parseStreamFormat(data)

	// 使用新的详细解析函数
	fmt.Println("\n使用详细的解析函数:")
	parseMongoDBPacket(data)

	// 详细解析数据包
	fmt.Println("\n详细解析数据包:")
	analysis := analyzeMongoPacket(data)
	fmt.Println(analysis)
}

// parseMongoDBPacket 按顺序解析MongoDB数据包的所有字段
func parseMongoDBPacket(data []byte) {
	if len(data) < 10 {
		fmt.Println("数据包太短，无法解析")
		return
	}

	offset := 0
	fmt.Println("\nMongoDB数据包字段解析:")
	fmt.Println("============================")

	// 1. 消息类型 (byte)
	messageType := data[offset]
	fmt.Printf("1. 消息类型: byte (偏移量:%d) = %d (protocol.MRsprowmsg)\n", offset, messageType)
	offset++

	// 2. 行数量 (uint32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析行数量")
		return
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	fmt.Printf("2. 行数量: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, rowCount)
	offset += 4

	// 3. gtid (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析GTID长度")
		return
	}
	gtidLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("3. gtid长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, gtidLen)
	offset += 4
	if offset+gtidLen > len(data) {
		fmt.Println("数据包不完整，无法读取GTID内容")
		return
	}
	gtid := string(data[offset : offset+gtidLen])
	fmt.Printf("   gtid内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+gtidLen-1, gtid)
	offset += gtidLen

	// 4. binlog_file (int32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析binlog文件")
		return
	}
	binlogFile := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("4. binlog_file: int32 (偏移量:%d-%d) = %d (MongoDB无binlog，固定值0)\n", offset, offset+3, binlogFile)
	offset += 4

	// 5. 数据类型 (int8)
	if offset >= len(data) {
		fmt.Println("数据包不完整，无法解析数据类型")
		return
	}
	dataType := int8(data[offset])
	var typeStr string
	switch dataType {
	case 13:
		typeStr = "插入(insert)"
	case 14:
		typeStr = "更新(update)"
	case 15:
		typeStr = "删除(delete)"
	default:
		typeStr = fmt.Sprintf("未知(%d)", dataType)
	}
	fmt.Printf("5. 数据类型: int8 (偏移量:%d) = %d (%s)\n", offset, dataType, typeStr)
	offset++

	// 6. db_name (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析数据库名长度")
		return
	}
	dbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("6. db_name长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, dbNameLen)
	offset += 4
	if offset+dbNameLen > len(data) {
		fmt.Println("数据包不完整，无法读取数据库名内容")
		return
	}
	dbName := string(data[offset : offset+dbNameLen])
	fmt.Printf("   db_name内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+dbNameLen-1, dbName)
	offset += dbNameLen

	// 7. table_name (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析表名长度")
		return
	}
	tbNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("7. table_name长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, tbNameLen)
	offset += 4
	if offset+tbNameLen > len(data) {
		fmt.Println("数据包不完整，无法读取表名内容")
		return
	}
	tbName := string(data[offset : offset+tbNameLen])
	fmt.Printf("   table_name内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+tbNameLen-1, tbName)
	offset += tbNameLen

	// 8. gtid再次发送 (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析第二个GTID长度")
		return
	}
	gtidLen2 := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("8. gtid长度(再次): uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, gtidLen2)
	offset += 4
	if offset+gtidLen2 > len(data) {
		fmt.Println("数据包不完整，无法读取第二个GTID内容")
		return
	}
	gtid2 := string(data[offset : offset+gtidLen2])
	fmt.Printf("   gtid内容(再次): string (偏移量:%d-%d) = '%s'\n", offset, offset+gtidLen2-1, gtid2)
	offset += gtidLen2

	// 9. sql (int32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析SQL值")
		return
	}
	sqlValue := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("9. sql: int32 (偏移量:%d-%d) = %d (MongoDB无SQL，固定值0)\n", offset, offset+3, sqlValue)
	offset += 4

	// 10. 列数量 (uint32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析列数量")
		return
	}
	columnNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("10. 列数量: uint32 (偏移量:%d-%d) = %d (MongoDB固定为3列)\n", offset, offset+3, columnNum)
	offset += 4

	// 11. 列名数组 (len+str[])
	fmt.Println("11. 列名数组: len+str[]")
	columnNames := make([]string, 0, columnNum)
	for i := 0; i < columnNum; i++ {
		if offset+4 > len(data) {
			fmt.Printf("   数据包不完整，无法解析第%d个列名长度\n", i+1)
			return
		}
		colNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
		fmt.Printf("   第%d列名长度: uint32 (偏移量:%d-%d) = %d\n", i+1, offset, offset+3, colNameLen)
		offset += 4

		if offset+colNameLen > len(data) {
			fmt.Printf("   数据包不完整，无法读取第%d个列名内容\n", i+1)
			return
		}
		colName := string(data[offset : offset+colNameLen])
		columnNames = append(columnNames, colName)
		fmt.Printf("   第%d列名内容: string (偏移量:%d-%d) = '%s'\n", i+1, offset, offset+colNameLen-1, colName)
		offset += colNameLen
	}

	// 12. 列属性数组 (uint32[])
	fmt.Println("12. 列属性数组: uint32[]")
	for i := 0; i < columnNum; i++ {
		if offset+4 > len(data) {
			fmt.Printf("   数据包不完整，无法解析第%d个列属性\n", i+1)
			return
		}
		colAttr := binary.LittleEndian.Uint32(data[offset : offset+4])
		fmt.Printf("   第%d列属性: uint32 (偏移量:%d-%d) = %d\n", i+1, offset, offset+3, colAttr)
		offset += 4
	}

	// 13. 行数 (uint32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析具体行数量")
		return
	}
	rowNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("13. 行数: uint32 (偏移量:%d-%d) = %d (每次发送1行数据)\n", offset, offset+3, rowNum)
	offset += 4

	// 14. 行内容 (len+str[])
	fmt.Println("14. 行内容: len+str[]")
	for i := 0; i < rowNum; i++ {
		fmt.Printf("   第%d行数据:\n", i+1)
		for j := 0; j < columnNum; j++ {
			if offset+4 > len(data) {
				fmt.Printf("      数据包不完整，无法解析第%d行第%d列的长度\n", i+1, j+1)
				return
			}
			valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
			fmt.Printf("      列'%s'长度: uint32 (偏移量:%d-%d) = %d\n",
				columnNames[j], offset, offset+3, valueLen)
			offset += 4

			if offset+valueLen > len(data) {
				fmt.Printf("      数据包不完整，无法读取第%d行第%d列的内容\n", i+1, j+1)
				return
			}
			value := string(data[offset : offset+valueLen])
			fmt.Printf("      列'%s'内容: string (偏移量:%d-%d) = '%s'\n",
				columnNames[j], offset, offset+valueLen-1, value)
			offset += valueLen
		}
	}

	// 解析旧值（仅用于UPDATE操作）
	if dataType == 14 && offset < len(data) {
		fmt.Println("15. 旧值数据 (仅用于UPDATE操作):")
		for i := 0; i < rowNum; i++ {
			fmt.Printf("   第%d行旧值:\n", i+1)
			for j := 0; j < columnNum; j++ {
				if offset+4 > len(data) {
					fmt.Printf("      数据包不完整，无法解析旧值第%d行第%d列的长度\n", i+1, j+1)
					return
				}
				valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
				fmt.Printf("      列'%s'旧值长度: uint32 (偏移量:%d-%d) = %d\n",
					columnNames[j], offset, offset+3, valueLen)
				offset += 4

				if offset+valueLen > len(data) {
					fmt.Printf("      数据包不完整，无法读取旧值第%d行第%d列的内容\n", i+1, j+1)
					return
				}
				value := string(data[offset : offset+valueLen])
				fmt.Printf("      列'%s'旧值内容: string (偏移量:%d-%d) = '%s'\n",
					columnNames[j], offset, offset+valueLen-1, value)
				offset += valueLen
			}
		}
	}

	// 添加剩余未解析的字节数提示
	if offset < len(data) {
		fmt.Printf("\n注意: 数据包中还有 %d 字节未解析\n", len(data)-offset)
	}
}
