package task

import (
	"consistent_sql/logger"
	"consistent_sql/model"
	"consistent_sql/protocol"
	"context"
	"database/sql"
	"encoding/json"
	"fmt"
	"net"
	"strconv"
	"sync"
	_ "td_driver_206/taosRestful"
	"time"
)

// TdTaskStatus 结构体用于跟踪任务状态
type TdTaskStatus struct {
	IsRunning bool
	LastTS    int64 // 使用时间戳替代MongoDB的ID
	Error     error
}

// TdTaskManager 结构体管理多个任务
type TdTaskManager struct {
	Tasks    map[string]context.CancelFunc
	Statuses map[string]*TdTaskStatus
	Mu       sync.Mutex

	// 连接池
	clients   map[string]*sql.DB
	clientsMu sync.Mutex
}

// SyncTdResult TDengine同步结果
type SyncTdResult struct {
	Data   []TDData
	LastTS int64
}

// TDData TDengine数据结构
type TDData struct {
	Timestamp int64                  `json:"ts"`
	Values    map[string]interface{} `json:"values"`
}

func NewTdTaskManager() *TdTaskManager {
	return &TdTaskManager{
		Tasks:    make(map[string]context.CancelFunc),
		Statuses: make(map[string]*TdTaskStatus),
		clients:  make(map[string]*sql.DB),
	}
}

// getTDClient 获取或创建TDengine连接
func (tm *TdTaskManager) getTDClient(url, user, password string) (*sql.DB, error) {
	tm.clientsMu.Lock()
	defer tm.clientsMu.Unlock()

	key := fmt.Sprintf("%s:%s:%s", url, user, password)
	if client, exists := tm.clients[key]; exists {
		return client, nil
	}

	// 构建连接字符串
	dsn := fmt.Sprintf("%s:%s@%s", user, password, url)
	db, err := sql.Open("taosSql", dsn)
	if err != nil {
		return nil, err
	}

	// 设置连接池参数
	db.SetMaxOpenConns(10)
	db.SetMaxIdleConns(5)
	db.SetConnMaxLifetime(time.Hour)

	// 测试连接
	if err := db.Ping(); err != nil {
		db.Close()
		return nil, err
	}

	tm.clients[key] = db
	return db, nil
}

// StartTaskTD 启动一个新的TDengine同步任务
func (tm *TdTaskManager) StartTaskTD(td model.RecvData, conn net.Conn) {
	tm.Mu.Lock()
	if _, exists := tm.Tasks[td.JobId]; exists {
		logger.Info("TDengine任务已存在: %s", td.JobId)
		tm.Mu.Unlock()
		protocol.SendError(conn, "1", "job id "+td.JobId+" exists", td.JobId)
		return
	}

	logger.Info("正在创建TDengine任务，ID=%s", td.JobId)

	// 解析开始时间
	startTime := parseStartTime(td.Gtid)

	tm.Statuses[td.JobId] = &TdTaskStatus{
		IsRunning: true,
		LastTS:    startTime,
		Error:     nil,
	}

	ctx, cancel := context.WithCancel(context.Background())
	tm.Tasks[td.JobId] = cancel
	tm.Mu.Unlock()

	// 启动同步任务
	go tm.syncTDTask(ctx, td, conn)
}

// syncTDTask 处理TDengine同步任务
func (tm *TdTaskManager) syncTDTask(ctx context.Context, td model.RecvData, conn net.Conn) {
	// 获取TDengine客户端
	db, err := tm.getTDClient(td.URL, td.DbUser, td.DbPasswd)
	if err != nil {
		tm.updateTaskStatus(td.JobId, 0, err)
		protocol.SendError(conn, "1", "连接TDengine失败: "+err.Error(), td.JobId)
		return
	}

	// 获取起始时间戳
	lastTS := tm.getTaskLastTS(td.JobId)

	// 发送开始消息
	protocol.SendError(conn, "0", "job "+td.JobId+" start", td.JobId)

	for {
		select {
		case <-ctx.Done():
			logger.Info("任务已取消: %s", td.JobId)
			tm.updateTaskStatus(td.JobId, lastTS, nil)
			return
		default:
			// 查询并同步数据
			result, err := tm.SyncTdTask(ctx, db, td.DbTable, lastTS)
			if err != nil {
				logger.Error("同步失败: %s, 错误: %v", td.JobId, err)
				tm.updateTaskStatus(td.JobId, lastTS, err)
				// 可以考虑在这里添加重试逻辑
				time.Sleep(5 * time.Second)
				continue
			}

			if len(result.Data) > 0 {
				// 发送同步结果
				tm.sendSyncResult(conn, td.JobId, &result, td.DbDatabase, td.DbTable)
				// 更新最后时间戳
				lastTS = result.LastTS
				tm.updateTaskStatus(td.JobId, lastTS, nil)
			} else {
				// 没有新数据，等待一段时间再查询
				time.Sleep(1 * time.Second)
			}
		}
	}
}

// updateTaskStatus 更新任务状态
func (tm *TdTaskManager) updateTaskStatus(jobId string, lastTS int64, err error) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if status, exists := tm.Statuses[jobId]; exists {
		if lastTS > 0 {
			status.LastTS = lastTS
		}
		status.Error = err
		if err != nil {
			status.IsRunning = false
		}
	}
}

// getTaskLastTS 获取任务的最后时间戳
func (tm *TdTaskManager) getTaskLastTS(jobId string) int64 {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if status, exists := tm.Statuses[jobId]; exists {
		return status.LastTS
	}
	return 0
}

// sendSyncResult 发送同步结果
func (tm *TdTaskManager) sendSyncResult(conn net.Conn, jobId string, result *model.SyncTdResult, dbName, tableName string) {
	if len(result.Data) == 0 {
		return
	}

	// 准备二进制流消息
	// [db_name长度 + db_name] + [tb_name长度 + tb_name] + [gtid] + [sql] +
	// [column_num] + [列名...] + [column_state] + [row_num] + [数据行...]

	// 构建消息
	var msg []byte

	// 数据库名
	dbNameBytes := []byte(dbName)
	dbNameLen := len(dbNameBytes)
	msg = append(msg, byte(dbNameLen&0xff), byte((dbNameLen>>8)&0xff), byte((dbNameLen>>16)&0xff), byte((dbNameLen>>24)&0xff))
	msg = append(msg, dbNameBytes...)

	// 表名
	tbNameBytes := []byte(tableName)
	tbNameLen := len(tbNameBytes)
	msg = append(msg, byte(tbNameLen&0xff), byte((tbNameLen>>8)&0xff), byte((tbNameLen>>16)&0xff), byte((tbNameLen>>24)&0xff))
	msg = append(msg, tbNameBytes...)

	// GTID (使用最后时间戳)
	gtid := fmt.Sprintf("%d", result.LastTS)
	gtidBytes := []byte(gtid)
	gtidLen := len(gtidBytes)
	msg = append(msg, byte(gtidLen&0xff), byte((gtidLen>>8)&0xff), byte((gtidLen>>16)&0xff), byte((gtidLen>>24)&0xff))
	msg = append(msg, gtidBytes...)

	// SQL (空)
	sqlBytes := []byte("")
	sqlLen := len(sqlBytes)
	msg = append(msg, byte(sqlLen&0xff), byte((sqlLen>>8)&0xff), byte((sqlLen>>16)&0xff), byte((sqlLen>>24)&0xff))
	msg = append(msg, sqlBytes...)

	// 定义列名 (至少包含ts, data, optype)
	columnNames := []string{"ts", "data", "optype"}

	// 列数量
	columnNum := len(columnNames)
	msg = append(msg, byte(columnNum&0xff), byte((columnNum>>8)&0xff), byte((columnNum>>16)&0xff), byte((columnNum>>24)&0xff))

	// 列名
	for _, colName := range columnNames {
		colBytes := []byte(colName)
		colLen := len(colBytes)
		msg = append(msg, byte(colLen&0xff), byte((colLen>>8)&0xff), byte((colLen>>16)&0xff), byte((colLen>>24)&0xff))
		msg = append(msg, colBytes...)
	}

	// 列属性 (使用0)
	msg = append(msg, 0, 0, 0, 0)

	// 行数量
	rowNum := len(result.Data)
	msg = append(msg, byte(rowNum&0xff), byte((rowNum>>8)&0xff), byte((rowNum>>16)&0xff), byte((rowNum>>24)&0xff))

	// 数据行
	for _, td := range result.Data {
		// 时间戳
		tsStr := fmt.Sprintf("%d", td.Ts)
		tsBytes := []byte(tsStr)
		tsLen := len(tsBytes)
		msg = append(msg, byte(tsLen&0xff), byte((tsLen>>8)&0xff), byte((tsLen>>16)&0xff), byte((tsLen>>24)&0xff))
		msg = append(msg, tsBytes...)

		// 数据
		var dataStr string
		if td.Data != nil {
			dataStr = fmt.Sprintf("%v", td.Data)
		} else {
			dataStr = ""
		}
		dataBytes := []byte(dataStr)
		dataLen := len(dataBytes)
		msg = append(msg, byte(dataLen&0xff), byte((dataLen>>8)&0xff), byte((dataLen>>16)&0xff), byte((dataLen>>24)&0xff))
		msg = append(msg, dataBytes...)

		// 操作类型
		optypeBytes := []byte(td.Optype)
		optypeLen := len(optypeBytes)
		msg = append(msg, byte(optypeLen&0xff), byte((optypeLen>>8)&0xff), byte((optypeLen>>16)&0xff), byte((optypeLen>>24)&0xff))
		msg = append(msg, optypeBytes...)
	}

	// 发送消息长度和消息内容
	msgLen := len(msg)
	header := []byte{byte(msgLen & 0xff), byte((msgLen >> 8) & 0xff), byte((msgLen >> 16) & 0xff), byte((msgLen >> 24) & 0xff)}

	// 先发送头部(长度)，再发送消息内容
	_, err := conn.Write(append(header, msg...))
	if err != nil {
		logger.Error("发送TD同步结果失败: %v", err)
	}
}

// parseStartTime 解析起始时间戳
func parseStartTime(gtid string) int64 {
	if ts, err := strconv.ParseInt(gtid, 10, 64); err == nil {
		return ts
	}
	return time.Now().Add(-time.Hour).UnixMilli() // 默认同步最近1小时的数据
}

// RemoveTask 从任务列表中删除任务
func (tm *TdTaskManager) RemoveTask(taskName string) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()
	delete(tm.Tasks, taskName)
	delete(tm.Statuses, taskName)
}

// StopTask 停止指定的任务
func (tm *TdTaskManager) StopTask(taskName string) bool {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if cancel, exists := tm.Tasks[taskName]; exists {
		cancel() // 取消任务
		delete(tm.Tasks, taskName)

		// 同时更新状态
		if status, ok := tm.Statuses[taskName]; ok {
			status.IsRunning = false
		}

		logger.Info("任务已停止: %s", taskName)
		return true
	} else {
		logger.Warn("任务不存在: %s", taskName)
		return false
	}
}

// StopAllTasks 停止所有任务
func (tm *TdTaskManager) StopAllTasks() {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	for taskName, cancel := range tm.Tasks {
		cancel() // 取消每个任务
		logger.Info("任务已停止: %s", taskName)
		delete(tm.Tasks, taskName)
	}
}

// GetAllTaskIDs 获取所有任务ID
func (tm *TdTaskManager) GetAllTaskIDs() []string {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	taskIDs := make([]string, 0, len(tm.Tasks))
	for id := range tm.Tasks {
		taskIDs = append(taskIDs, id)
	}
	return taskIDs
}

// GetTaskStatus 查询指定任务的状态
func (tm *TdTaskManager) GetTaskStatus(taskName string) (*TdTaskStatus, bool) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	status, exists := tm.Statuses[taskName]
	return status, exists
}

// SyncTdTask 查询并同步TDengine数据
func (tm *TdTaskManager) SyncTdTask(ctx context.Context, db *sql.DB, tableName string, startTS int64) (model.SyncTdResult, error) {
	// 构建查询SQL
	var query string
	if startTS > 0 {
		query = fmt.Sprintf("SELECT * FROM %s WHERE _ts > %d ORDER BY _ts ASC LIMIT 100", tableName, startTS)
	} else {
		query = fmt.Sprintf("SELECT * FROM %s ORDER BY _ts ASC LIMIT 100", tableName)
	}

	// 执行查询
	rows, err := db.QueryContext(ctx, query)
	if err != nil {
		return model.SyncTdResult{}, err
	}
	defer rows.Close()

	// 获取列信息
	columns, err := rows.Columns()
	if err != nil {
		return model.SyncTdResult{}, err
	}

	// 准备结果
	var result model.SyncTdResult
	result.Data = []model.TDSourceDBData{}
	var lastTS int64

	// 遍历结果
	for rows.Next() {
		// 准备接收数据的变量
		scanArgs := make([]interface{}, len(columns))
		values := make([]interface{}, len(columns))
		for i := range values {
			scanArgs[i] = &values[i]
		}

		// 读取当前行
		if err := rows.Scan(scanArgs...); err != nil {
			logger.Error("读取数据行失败: %s", err.Error())
			continue
		}

		// 构建数据项
		timestamp := int64(0)
		dataMap := make(map[string]interface{})

		for i, col := range columns {
			if col == "_ts" {
				// 处理时间戳
				if v, ok := values[i].(time.Time); ok {
					timestamp = v.Unix() * 1000 // 转换为毫秒
				} else if v, ok := values[i].(int64); ok {
					timestamp = v
				}
			} else {
				// 处理其他字段
				dataMap[col] = values[i]
			}
		}

		// 更新最后时间戳
		if timestamp > lastTS {
			lastTS = timestamp
		}

		// 转换数据为JSON
		dataJSON, _ := json.Marshal(dataMap)

		// 创建数据项
		data := model.TDSourceDBData{
			Ts:     timestamp,
			Data:   string(dataJSON),
			Optype: "'insert'",
		}

		result.Data = append(result.Data, data)
	}

	// 检查遍历过程中是否有错误
	if err := rows.Err(); err != nil {
		return result, err
	}

	result.LastTS = lastTS
	return result, nil
}

// GetTableColumns 获取TDengine表的列信息
func (tm *TdTaskManager) GetTableColumns(db *sql.DB, tableName string) ([]string, error) {
	// 查询表结构
	query := fmt.Sprintf("DESCRIBE %s", tableName)
	rows, err := db.Query(query)
	if err != nil {
		return nil, err
	}
	defer rows.Close()

	// 获取列名
	var columns []string
	for rows.Next() {
		var field, dataType, length, note string
		if err := rows.Scan(&field, &dataType, &length, &note); err != nil {
			return nil, err
		}
		columns = append(columns, field)
	}

	if err := rows.Err(); err != nil {
		return nil, err
	}

	return columns, nil
}

// Stringtime 将整数时间戳转换为可读时间
func (tm *TdTaskManager) Stringtime(IntTime int64) string {
	if IntTime == 0 {
		return ""
	}
	// 转换为时间格式
	t := time.Unix(IntTime/1000, 0) // 假设时间戳为毫秒级
	// 返回格式化时间
	return t.Format("2006-01-02 15:04:05")
}
