<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlarmRecord extends CI_Controller
{
    public $config;
    public $key;
    public $db;
    public $Login_model;
    public $Cluster_model;

    public function __construct()
    {
        parent::__construct();
        header('Access-Control-Allow-Origin:*'); // *代表允许任何网址请求
        header('Access-Control-Allow-Headers: Content-Type,Content-Length,Accept-Encoding,X-Requested-with, Origin'); // 设置允许自定义请求头的字段
        header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE'); // 允许请求的类型
        header('Access-Control-Allow-Headers:x-requested-with,content-type,Token'); //允许接受token
        header('Content-Type: application/json;charset=utf-8');
        
        // 加载必要的模型和库
        $this->load->model('Login_model');
        $this->load->model('Cluster_model');
        
        // 加载配置
        $this->config->load('myconfig');
        $this->key = $this->config->item('key');
    }

    /**
     * 获取告警记录列表
     */
    public function getAlarmRecordList()
    {
        try {
            // 获取查询参数
            $pageNo = isset($_GET['pageNo']) ? intval($_GET['pageNo']) : 1;
            $pageSize = isset($_GET['pageSize']) ? intval($_GET['pageSize']) : 10;
            $job_type = isset($_GET['job_type']) ? $_GET['job_type'] : '';
            $status = isset($_GET['status']) ? $_GET['status'] : '';
            $alarm_level = isset($_GET['alarm_level']) ? $_GET['alarm_level'] : '';
            
            // 构建查询条件
            $where = [];
            $params = [];
            
            if (!empty($job_type)) {
                $where[] = "alarm_type = ?";
                $params[] = $job_type;
            }
            
            if (!empty($status)) {
                $where[] = "status = ?";
                $params[] = $status;
            }
            
            if (!empty($alarm_level)) {
                $where[] = "alarm_level = ?";
                $params[] = $alarm_level;
            }
            
            $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
            
            // 查询总数
            $countSql = "SELECT COUNT(*) as total FROM cluster_alarm_info $whereClause";
            $countResult = $this->Cluster_model->getList($countSql, $params);
            $total = $countResult[0]['total'] ?? 0;
            
            // 查询列表数据
            $offset = ($pageNo - 1) * $pageSize;
            $listSql = "SELECT 
                            id, 
                            job_id, 
                            alarm_level, 
                            alarm_type, 
                            status, 
                            occur_timestamp, 
                            job_info, 
                            handle_time, 
                            handler_name, 
                            cluster_id, 
                            shardid, 
                            compnodeid 
                        FROM cluster_alarm_info 
                        $whereClause 
                        ORDER BY occur_timestamp DESC 
                        LIMIT ? OFFSET ?";
            
            $listParams = array_merge($params, [$pageSize, $offset]);
            $list = $this->Cluster_model->getList($listSql, $listParams);
            
            // 处理列表数据
            $formattedList = [];
            foreach ($list as $item) {
                $jobInfo = json_decode($item['job_info'], true);
                
                // 提取推送类型
                $pushTypes = $this->getPushTypes($item['alarm_type']);
                
                $formattedItem = [
                    'id' => $item['id'],
                    'alarm_type' => $item['alarm_type'],
                    'alarm_level' => $item['alarm_level'],
                    'status' => $item['status'],
                    'occur_timestamp' => $item['occur_timestamp'],
                    'handle_time' => $item['handle_time'],
                    'handler_name' => $item['handler_name'],
                    'object' => $jobInfo['ip'] ?? ($jobInfo['message'] ?? ''),
                    'push_type' => $pushTypes
                ];
                
                $formattedList[] = $formattedItem;
            }
            
            // 返回结果
            $result = [
                'code' => 200,
                'data' => [
                    'list' => $formattedList,
                    'total' => $total,
                    'pageNo' => $pageNo,
                    'pageSize' => $pageSize
                ]
            ];
            
            echo json_encode($result);
            
        } catch (Exception $e) {
            log_message('error', 'Error in getAlarmRecordList: ' . $e->getMessage());
            $result = [
                'code' => 500,
                'message' => $e->getMessage()
            ];
            echo json_encode($result);
        }
    }
    
    /**
     * 获取告警类型对应的推送方式
     */
    private function getPushTypes($alarmType)
    {
        $sql = "SELECT push_type FROM cluster_alarm_user WHERE id = ?";
        $result = $this->Cluster_model->getList($sql, [$alarmType]);
        
        if (!empty($result) && !empty($result[0]['push_type'])) {
            return explode(',', $result[0]['push_type']);
        }
        
        return [];
    }
    
    /**
     * 处理告警
     */
    public function handleAlarm()
    {
        try {
            $post = json_decode(file_get_contents('php://input'), true);
            $id = $post['id'] ?? 0;
            $status = $post['status'] ?? '';
            $user_name = $post['user_name'] ?? '';
            
            if (empty($id) || empty($status) || empty($user_name)) {
                throw new Exception('参数不完整');
            }
            
            $updateSql = "UPDATE cluster_alarm_info SET status = ?, handler_name = ?, handle_time = NOW() WHERE id = ?";
            $result = $this->Cluster_model->updateList($updateSql, [$status, $user_name, $id]);
            
            if ($result) {
                $response = [
                    'code' => 200,
                    'message' => '处理成功'
                ];
            } else {
                $response = [
                    'code' => 500,
                    'message' => '处理失败'
                ];
            }
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            log_message('error', 'Error in handleAlarm: ' . $e->getMessage());
            $response = [
                'code' => 500,
                'message' => $e->getMessage()
            ];
            echo json_encode($response);
        }
    }
    
    /**
     * 获取告警详情
     */
    public function getAlarmDetail()
    {
        try {
            $id = isset($_GET['id']) ? $_GET['id'] : 0;
            
            if (empty($id)) {
                throw new Exception('参数不完整');
            }
            
            $sql = "SELECT * FROM cluster_alarm_info WHERE id = ?";
            $result = $this->Cluster_model->getList($sql, [$id]);
            
            if (empty($result)) {
                throw new Exception('告警记录不存在');
            }
            
            $alarm = $result[0];
            $jobInfo = json_decode($alarm['job_info'], true);
            
            // 获取集群和分片信息
            $clusterName = '';
            $shardName = '';
            
            if (!empty($alarm['cluster_id'])) {
                $clusterSql = "SELECT name FROM db_clusters WHERE id = ?";
                $clusterResult = $this->Cluster_model->getList($clusterSql, [$alarm['cluster_id']]);
                if (!empty($clusterResult)) {
                    $clusterName = $clusterResult[0]['name'];
                }
                
                if (!empty($alarm['shardid'])) {
                    $shardSql = "SELECT name FROM shards WHERE id = ? AND db_cluster_id = ?";
                    $shardResult = $this->Cluster_model->getList($shardSql, [$alarm['shardid'], $alarm['cluster_id']]);
                    if (!empty($shardResult)) {
                        $shardName = $shardResult[0]['name'];
                    }
                }
            }
            
            // 获取推送记录
            $pushSql = "SELECT * FROM cluster_alarm_push_log WHERE alarm_type = ? ORDER BY create_at DESC LIMIT 10";
            $pushResult = $this->Cluster_model->getList($pushSql, [$alarm['alarm_type']]);
            
            $detail = [
                'id' => $alarm['id'],
                'alarm_type' => $alarm['alarm_type'],
                'alarm_level' => $alarm['alarm_level'],
                'status' => $alarm['status'],
                'occur_timestamp' => $alarm['occur_timestamp'],
                'handle_time' => $alarm['handle_time'],
                'handler_name' => $alarm['handler_name'],
                'cluster_name' => $clusterName,
                'shard_name' => $shardName,
                'job_info' => $jobInfo,
                'push_records' => $pushResult
            ];
            
            $response = [
                'code' => 200,
                'data' => $detail
            ];
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            log_message('error', 'Error in getAlarmDetail: ' . $e->getMessage());
            $response = [
                'code' => 500,
                'message' => $e->getMessage()
            ];
            echo json_encode($response);
        }
    }
} 