<?php
defined('BASEPATH') or exit('No direct script access allowed');

class CdcConfig extends MY_Controller
{
	public $Cluster_model;
	public $Login_model;
	public $Cdc_model;
	public $MongoDB_model;
	public $db;
	protected $alarm_service;

	public function __construct()
	{
		parent::__construct();

		header('Access-Control-Allow-Origin:*');
		header('Access-Control-Allow-Headers: Content-Type,Content-Length,Accept-Encoding,X-Requested-with, Origin');
		header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE');
		header('Access-Control-Allow-Headers:x-requested-with,content-type,Token');
		header('Content-Type: application/json;charset=utf-8');

		// 加载模型和服务
		$this->load->model('Cluster_model');
		$this->load->model('Login_model');
		$this->load->model('Cdc_model');
		$this->load->model('mongodb_model', 'MongoDB_model');  // 别名加载 MongoDB 模型
		$this->db = $this->load->database('default', true);
		$this->load->library('AlarmService');
		$this->alarm_service = new AlarmService();
	}

	/**
	 * 执行CURL请求
	 *
	 * @param string $url 请求的URL
	 * @param array $postData 请求数据
	 * @param int $timeout 超时时间，默认为0（无限制）
	 * @return string|bool 请求结果或false（如果请求失败）
	 */
	protected function performCurlRequest($url, $postData, $timeout = 5)
	{
		log_message('debug', '执行CURL请求 - URL: ' . $url);
		log_message('debug', '请求数据: ' . json_encode($postData));
		log_message('debug', '===performCurlRequest - 开始执行CURL请求，URL: ' . $url . ', 超时设置: ' . $timeout . '秒');

		$curl = curl_init();
		curl_setopt_array(
			$curl,
			array(
				CURLOPT_URL => $url,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING => '',
				CURLOPT_MAXREDIRS => 10,
				CURLOPT_TIMEOUT => $timeout,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST => 'POST',
				CURLOPT_POSTFIELDS => json_encode($postData),
				CURLOPT_HTTPHEADER => array(
					'Content-Type: application/json'
				),
			)
		);

		$response = curl_exec($curl);

		if (!$response) {
			$curlError = curl_error($curl);
			$curlErrno = curl_errno($curl);
			log_message('error', 'CURL请求失败: ' . $curlError . ', 错误码: ' . $curlErrno);
			log_message('debug', '===performCurlRequest - CURL请求失败，错误: ' . $curlError . ', 错误码: ' . $curlErrno);
			curl_close($curl);
			return false;
		}

		$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$totalTime = curl_getinfo($curl, CURLINFO_TOTAL_TIME);
		log_message('debug', '===performCurlRequest - CURL请求成功，HTTP状态码: ' . $httpCode . ', 总耗时: ' . $totalTime . '秒');
		log_message('debug', 'CURL请求成功，返回数据: ' . $response);
		curl_close($curl);

		return $response;
	}

	/**
	 * 统一的JSON响应方法
	 */
	protected function jsonResponse($data = [], $code = 200, $message = 'Success')
	{
		$response = [
			'code' => $code,
			'message' => $message
		];

		if (!empty($data)) {
			$response['data'] = $data;
		}

		header('Content-Type: application/json; charset=utf-8');
		echo json_encode($response);
		return;
	}

	/**
	 * Token验证方法
	 */
	protected function validateToken($token)
	{
		if (empty($token)) {
			throw new Exception('Token不能为空');
		}

		$res_token = $this->Login_model->getToken($token, 'D', config_item('key'));
		if (empty($res_token)) {
			throw new Exception('Token验证失败');
		}

		// 验证用户是否存在
		$sql = "SELECT count(id) as count FROM kunlun_user WHERE name = ?";
		$res = $this->Login_model->getList($sql, [$res_token]);
		if (empty($res) || $res[0]['count'] == 0) {
			throw new Exception('用户不存在');
		}

		return $res_token;
	}

	/**
	 * 获取并验证请求头中的Token
	 */
	protected function getRequestToken()
	{
		$headers = apache_request_headers();
		return isset($headers['Token']) ? $headers['Token'] : '';
	}

	public function getConfig()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			$sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_message_config";
			$config = $this->Cluster_model->getList($sql);

			return $this->jsonResponse([
				'list' => $config ?: [],
				'total' => count($config)
			]);
		} catch (Exception $e) {
			log_message('error', 'Get alarm config failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}


	public function getCdcList()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);


			// 调用封装的方法获取和更新leader节点
			$this->updateAndCheckCdcLeader();

			$string = json_decode(@file_get_contents('php://input'), true);
			if (isset($string['name']) && $string['name'] != "") {
				$sql = "select * from cluster_cdc_server where group_name='" . $string['name'] . "' order by group_name asc ";
			} else {
				$sql = "select * from cluster_cdc_server order by group_name asc ";
			}
			$res = $this->Cluster_model->getList($sql);

			$groupSql = "select * from cluster_cdc_server group by group_name";
			$groupData = $this->Cluster_model->getList($groupSql);

			return $this->jsonResponse([
				'list' => $res,
				'group' => $groupData,
				'total' => count($res)
			]);
		} catch (Exception $e) {
			log_message('error', 'Get CDC list failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	public function getCdcCreatedInfo($cdc_job_id)
	{

		$updata_sql = "select memo,`status`,`job_info` from cluster_general_job_log  where `job_type`=? and `job_id`=? order by id desc limit 1";


		$res = $this->Cluster_model->getList($updata_sql, ['add_cdc_worker_list', $cdc_job_id]);

		//pa([$updata_sql,['add_cdc_worker_list', $cdc_job_id],$res]);

		$res[0]['job_info'] = json_decode($res[0]['job_info'], true);

		foreach ($res[0]['job_info']['paras']['output_plugins'] as $key => &$value) {
			$value['plugin_param'] = json_decode($value['plugin_param'], true);
		}
		unset($value);
		return $res;
	}


	public function getCdcWorkerList()
	{
		//获取token
		$arr = apache_request_headers(); //获取请求头数组

		$this->load->model('Cluster_model');


		$string = json_decode(@file_get_contents('php://input'), true);
		$job_id = $string['job_id'];
		$job_name = $string['job_name'];
		$is_delete = $string['is_delete'];

		if ($is_delete == "2") {
			if ($job_name != "") {
				$sql = "select * from cluster_cdc_worker where (`job_name` like ? or worker_param like ? ) and `delete_time` is not null and `job_name` is not null and `job_name` != '' order by status desc ";
				$params = ["%{$job_name}%", "%{$job_name}%"];
			} else {
				$sql = "select * from cluster_cdc_worker where `delete_time` is not null and `job_id` is not null and `job_name` is not null and `job_name` != '' order by status desc ";
				$params = [];
			}
		} else {
			if ($job_name != "") {
				$sql = "select * from cluster_cdc_worker where (`job_name` like ? or worker_param like ? )  and `delete_time` is  null and `worker_param` is not null and `job_name` is not null and `job_name` != '' order by status desc ";
				$params = ["%{$job_name}%", "%{$job_name}%"];
			} else {
				$sql = "select * from cluster_cdc_worker where `delete_time` is  null and `job_id` is not null and `worker_param` is not null and `job_name` is not null and `job_name` != '' order by status desc ";
				$params = [];
			}
		}

		$res = $this->Cluster_model->getList($sql, $params);

		if ($res) {
			foreach ($res as $key => &$v) {


				$v['worker_param'] = json_decode($v['worker_param'], true);
				$v['stream_input'] = $this->getStreamBySrc($v['job_id']);
				$v['stream_output'] = $this->getStreamByDst($v['job_id']);

				foreach ($v['worker_param']['output_plugins'] as $k => &$v1) {
					$v1['plugin_param'] = json_decode(urldecode($v1['plugin_param']), true);
				}

				$v['cdc_src_created_info'] = $this->getCdcCreatedInfo($v['job_id']);
				$v['hot_switch'] = $this->getHotSwitch($v['job_id']);
				$v['cache_sql_num'] = "0";
				$v['pending_sql_num'] = "0";

				$v['cluster_info'] = $this->getClusterInfo($v['db_cluster_id']);


			}
		}
		unset($v);

		if (!$res) {
			$res = [];
		}
		$resp = [
			'code' => 200,
			'list' => $res,
			'total' => count($res)
		];
		print_r(json_encode($resp));
	}


	public function getStreamBySrc($src_id)
	{
		$this->load->model('Cluster_model');
		$sql = "select * from cluster_steams where cdc_src=?";
		return $this->Cluster_model->getList($sql, [$src_id]);
	}

	public function getStreamByDst($dst_id)
	{
		$this->load->model('Cluster_model');
		$sql = "select * from cluster_steams where cdc_dst=?";
		return $this->Cluster_model->getList($sql, [$dst_id]);
	}


	public function getCdcWorkerStatus()
	{
		$arr = apache_request_headers(); //获取请求头数组
		$token = $arr["Token"];
		if (empty($token)) {
			$data['code'] = 201;
			$data['message'] = 'token不能为空';
			print_r(json_encode($data));
			return;
		}
		$string = json_decode(@file_get_contents('php://input'), true);
		$this->load->model('Cdc_model');

		$group_name = $string['group_name'];
		$job_id = $string['job_id'];

		if ($group_name == "" || $job_id == "") {
			$resp = [
				'code' => 201,
				'message' => 'group_name or job_id is empty'
			];
			print_r(json_encode($resp));
			return;
		}

		$resp = $this->Cdc_model->get_cdc_worker_state($string['group_name'], $string['job_id']);
		print_r(json_encode($resp));
	}


	/**
	 * @return mixed
	 */
	public function getHotSwitch($_job_id)
	{

		$this->load->model('Cluster_model');
		$sql = "select * from cluster_cdc_hot_switch where cdc_job_id=? order by id desc limit 1";
		$res = $this->Cluster_model->getList($sql, $_job_id);
		return $res;
	}


	public function getClusterInfo($db_cluster_id)
	{
		$this->load->model('Cluster_model');
		$sql = "select id,name,nick_name from db_clusters where id=?";
		$res = $this->Cluster_model->getList($sql, $db_cluster_id);
		return $res[0];
	}


	public function GetCurlData($http, $post)
	{

		// 执行第二个curl请求
		$secondRs = $this->performCurlRequest($http, $post);

		// 检查第二个curl请求是否成功
		if (!$secondRs) {
			log_message('debug', '===GetCurlData - 第二阶段失败: 无法从leader节点获取响应');
			return false;
		}

		log_message('debug', '===GetCurlData - 第二阶段成功，获取到完整响应');

		// 检查响应中是否包含超时错误信息
		if (strpos($secondRs, 'Max connect timeout') !== false || strpos($secondRs, 'timeout') !== false) {
			log_message('warning', '===GetCurlData - 响应中检测到超时相关错误: ' . substr($secondRs, 0, 200) . '...');
		}

		return $secondRs;
	}


	public function DeleteHardCdcWorker()
	{
		//获取token
		$arr = apache_request_headers(); //获取请求头数组
		$token = $arr["Token"];
		if (empty($token)) {
			$data['code'] = 201;
			$data['message'] = 'token不能为空';
			print_r(json_encode($data));
			return;
		}
		$this->load->model('Cluster_model');
		//判断参数
		$string = json_decode(@file_get_contents('php://input'), true);

		//查询cdc job_id 是否和stream 有关联

		$stram_sql = "delete FROM cluster_cdc_worker WHERE id=? ";
		$streamList = $this->Cluster_model->updateList($stram_sql, [$string['id']]);
		if ($streamList) {
			$data['code'] = 201;
			$data['message'] = '该job_id和stream有关联，请先删除stream';
			print_r(json_encode($data));
			return;
		}

		$data['code'] = 200;
		$data['message'] = 'ok';
		print_r(json_encode($data));
		return;
	}

	//获取CDC 同步状态列表
	public function GetCdcWorkerStageList()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取GET参数
			$request = $_GET;
			$pageNo = isset($request['pageNo']) ? $request['pageNo'] : 1;
			$pageSize = isset($request['pageSize']) ? $request['pageSize'] : 10;
			$job_id = isset($request['job_id']) ? $request['job_id'] : '';
			$start = ($pageNo - 1) * $pageSize;

			$group_name = isset($request['group_name']) ? $request['group_name'] : '';
			$start_time = isset($request['start']) ? $request['start'] : '';
			$end_time = isset($request['end']) ? $request['end'] : '';

			// 如果未提供开始时间，默认为30天前
			if (empty($start_time)) {
				$start_time = date("Y-m-d H:i:s", time() - (86400 * 30));
			}

			// 如果未提供结束时间，默认为当前时间
			if (empty($end_time)) {
				$end_time = date("Y-m-d H:i:s", time());
			}

			log_message('debug', '获取CDC Worker阶段列表开始 - job_id: ' . $job_id . ', 时间范围: ' . $start_time . ' 至 ' . $end_time);

			// 安全处理job_id参数
			$job_ids = [];
			if (!empty($job_id)) {
				$job_ids = explode(',', $job_id);
				$job_ids = array_map('trim', $job_ids);
				$job_ids = array_filter($job_ids);
			}

			if (empty($job_ids)) {
				log_message('warning', 'GetCdcWorkerStageList - 未提供有效的job_id参数');
				return $this->jsonResponse([], 200, '未提供有效的job_id参数');
			}

			// 构建安全的参数化查询
			$placeholders = implode(',', array_fill(0, count($job_ids), '?'));
			$select_sql = "SELECT * FROM cdc_job_state WHERE job_id IN($placeholders)";

			// 添加时间范围过滤条件
			$select_sql .= " AND time BETWEEN UNIX_TIMESTAMP(?) AND UNIX_TIMESTAMP(?)";
			$params = array_merge($job_ids, [$start_time, $end_time]);

			// 按时间排序
			$select_sql .= " ORDER BY time DESC";

			// 如果指定了分页大小，添加LIMIT子句
			if ($pageSize > 0) {
				$select_sql .= " LIMIT ?, ?";
				$params[] = (int)$start;
				$params[] = (int)$pageSize;
			}

			log_message('debug', 'GetCdcWorkerStageList - SQL: ' . $select_sql);
			log_message('debug', 'GetCdcWorkerStageList - 参数: ' . json_encode($params));

			$Res = $this->Cluster_model->getList($select_sql, $params);

			if ($Res === false) {
				log_message('warning', 'GetCdcWorkerStageList - 查询结果为空');
				$Res = [];
			}

			// 处理结果集
			foreach ($Res as $k => $vs) {
				$param_data = json_decode($vs['param'], true);
				if (json_last_error() !== JSON_ERROR_NONE) {
					log_message('warning', 'GetCdcWorkerStageList - 解析param字段失败: ' . json_last_error_msg());
					$param_data = [];
				}

				$Res[$k]['param'] = $param_data;
				$Res[$k]['time'] = date("Y-m-d H:i:s", $vs['time']);
			}

			// 获取记录总数（不含分页限制）
			$count_params = array_merge($job_ids, [$start_time, $end_time]);
			$total_sql = "SELECT COUNT(*) AS count FROM cdc_job_state WHERE job_id IN($placeholders) AND time BETWEEN UNIX_TIMESTAMP(?) AND UNIX_TIMESTAMP(?)";
			$res_total = $this->Cluster_model->getList($total_sql, $count_params);

			$total_count = ($res_total && isset($res_total[0]['count'])) ? (int)$res_total[0]['count'] : 0;
			log_message('debug', 'GetCdcWorkerStageList - 总记录数: ' . $total_count);

			return $this->jsonResponse([
				'list' => $Res,
				'total' => $total_count
			]);
		} catch (Exception $e) {
			log_message('error', 'Get CDC worker stage list failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	public function DeleteCdcWorker()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 解析请求数据
			$string = json_decode(@file_get_contents('php://input'), true);

			// 查询cdc job_id 是否和stream 有关联
			$stram_sql = "SELECT * FROM cluster_steams WHERE (cdc_src=? or cdc_dst=?) and delete_at is null ";
			$streamList = $this->Cluster_model->getList($stram_sql, [$string['job_id'], $string['job_id']]);

			if ($streamList) {
				return $this->jsonResponse([], 201, '该job_id和stream有关联，请先删除stream');
			}

			// 获取cdc主节点信息
			$master_sql = "SELECT a.*,b.* FROM cluster_cdc_worker a LEFT JOIN cluster_cdc_server b ON a.group_name=b.group_name WHERE a.job_id= ? AND a.group_name=?";
			$params = [
				$string['job_id'],
				$string['group_name']
			];
			$masterList = $this->Cluster_model->getList($master_sql, $params);

			foreach ($masterList as $key => $value) {
				if ($value['worker_param'] == '') {
					$update_sql = "update cluster_cdc_worker set delete_time=? where job_id=? and group_name=?";
					$this->Cluster_model->updateList($update_sql, [time(), $value['job_id'], $value['group_name']]);
					return $this->jsonResponse([
						'list' => [
							'job_id' => ''
						]
					], 200, 'ok');
				}

				$job = json_decode($value['worker_param'], true);

				if (!empty($job)) {
					$curlData = [
						"version" => "1.0",
						"job_id" => "",
						"job_type" => "del_dump_table",
						"timestamp" => time(),
						"user_name" => "kunlun_test",
						"paras" => [
							"meta_db" => $job['metadb'],
							"cluster_name" => $job['cluster_name'],
							"dump_tables" => $job['dump_tables']
						]
					];

					$responseArr = $this->Cdc_model->delete_cdc_worker_task($string['job_id'], $string['group_name'], $curlData);

					if (!$responseArr) {
						return $this->jsonResponse([], 201, '删除失败');
					}

					if ($responseArr['status'] == 'Fail') {
						return $this->jsonResponse(['do' => $string['job_status']], 201, $responseArr['error_info']);
					} else {
						return $this->jsonResponse([
							'list' => $responseArr,
							'do' => $string['job_status']
						], 200, '添加成功');
					}
				}
			}

			// 如果循环结束仍未返回，说明没有找到匹配的记录
			return $this->jsonResponse([], 201, '未找到匹配的CDC Worker记录');
		} catch (Exception $e) {
			log_message('error', 'Delete CDC worker failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	public function CdcDelete()
	{
		//获取token
		$arr = apache_request_headers(); //获取请求头数组
		$token = $arr["Token"];
		if (empty($token)) {
			$data['code'] = 201;
			$data['message'] = 'token不能为空';
			print_r(json_encode($data));
			return;
		}
		$this->load->model('Cluster_model');
		//判断参数
		$string = json_decode(@file_get_contents('php://input'), true);
		$this->load->model('Cluster_model');
		$cdc_id = $string['paras']['name'];

		$delete_sql = "delete from cluster_cdc_server where id=" . $cdc_id;
		$Res = $this->Cluster_model->updateList($delete_sql);

		$status = $Res == 0 ? 'failed' : 'done';
		$add_log = "INSERT INTO `cluster_general_job_log` ( `job_id`, `related_id`, `job_type`, `status`, `memo`, `when_started`, `when_ended`, `job_info`, `user_name`) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?)";

		$params = [
			'job_id' => '',
			'related_id' => '',
			'job_type' => 'delete_cdc_list',
			'status' => $status,
			'memo' => $Res,
			'when_started' => date("Y-m-d H:i:s", time()),
			'when_ended' => date("Y-m-d H:i:s", time()),
			'job_info' => json_encode($string),
			'user_name' => ''
		];

		$this->Cluster_model->updateList($add_log, $params);

		if ($Res > 0) {
			$data['code'] = 200;
			$data['message'] = '成功';
		} else {
			$data['code'] = 201;
			$data['message'] = $Res;
		}
		print_r(json_encode($data));
		return;
	}


	public function getCdcGroupInfo()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 修改查询语句，只获取 master = 1 的记录
			$sql = "SELECT * FROM cluster_cdc_server WHERE `master` = 1 GROUP BY group_name";
			$masterList = $this->Cluster_model->getList($sql);

			if ($masterList) {
				$curlData = [
					"version" => "1.0",
					"job_id" => "",
					"job_type" => "list_support_plugins",
					"timestamp" => time(),
					"user_name" => "kunlun_test"
				];

				// 获取插件支持列表
				foreach ($masterList as $key => $value) {
					$curl_addr = "http://" . trim($masterList[$key]['host_addr']) . ":" . trim($masterList[$key]['port']) . "/kunlun_cdc";
					$response = $this->GetCurlData($curl_addr, $curlData);

					// 获取成功后cdc返回主节点数据
					if ($response) {
						$masterList[$key]['list_dumps'] = json_decode($response, true);
					} else {
						$masterList[$key]['list_dumps'] = $response;
					}
				}

				// 获取配置
				$curlData = [
					"version" => "1.0",
					"job_id" => "",
					"job_type" => "list_cdc_conf",
					"timestamp" => time(),
					"user_name" => "kunlun_test"
				];

				foreach ($masterList as $key => $value) {
					$curl_addr = "http://" . trim($masterList[$key]['host_addr']) . ":" . trim($masterList[$key]['port']) . "/kunlun_cdc";
					$response = $this->GetCurlData($curl_addr, $curlData);

					if ($response) {
						$masterList[$key]['conf'] = json_decode($response, true);
					} else {
						$masterList[$key]['conf'] = $response;
					}
				}
			}

			return $this->jsonResponse($masterList ?: []);
		} catch (Exception $e) {
			log_message('error', 'Get CDC group info failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}


	public function cdcEdit()
	{
		//获取token
		$arr = apache_request_headers(); //获取请求头数组
		$token = $arr["Token"];
		if (empty($token)) {
			$data['code'] = 201;
			$data['message'] = 'token不能为空';
			print_r(json_encode($data));
			return;
		}
		$this->load->model('Cluster_model');
		//判断参数
		$string = json_decode(@file_get_contents('php://input'), true);
		$stringArr = $string['paras']['temp'];
		$this->load->model('Cluster_model');
		foreach ($stringArr as $key => $value) {
			$sql = "INSERT INTO cluster_cdc_server (`host_addr`, `port`, `master`, `create_time`, `status`,`group_name`) VALUES (?,?,?,?,?,?)";
			$params = [
				$value['hostaddr'],
				$value['port'],
				0,
				time(),
				1,
				$string['cdc_group_name']
			];

			$this->Cluster_model->updateList($sql, $params);
		}
		$data['code'] = 200;
		$data['message'] = '获取CDC 服务成功';
		print_r(json_encode($data));
		return;
	}

	/**
	 * 更新并检查CDC集群中的leader节点信息，同时检测主备切换
	 *
	 * @return array 离线的节点组列表
	 */
	protected function updateAndCheckCdcLeader()
	{
		log_message('debug', '开始获取CDC集群leader节点并检测主备切换');

		// 获取当前主节点记录，用于后续检测主备切换
		$previous_leaders = [];
		$prev_leaders_sql = "SELECT group_name, host_addr, port FROM cluster_cdc_server WHERE `master` = 1";
		$prev_leaders_result = $this->Cluster_model->getList($prev_leaders_sql);

		if ($prev_leaders_result) {
			foreach ($prev_leaders_result as $leader) {
				// Trim host_addr and port when reading from DB
				$prev_host = isset($leader['host_addr']) ? trim($leader['host_addr']) : '';
				$prev_port = isset($leader['port']) ? trim($leader['port']) : '';
				if (!empty($prev_host) && !empty($prev_port)) {
					$previous_leaders[$leader['group_name']] = $prev_host . ':' . $prev_port;
				}
			}
		}
		log_message('debug', 'Previous leaders fetched: ' . json_encode($previous_leaders));

		// 重置所有节点状态
		$updatable = "UPDATE cluster_cdc_server SET `master`=0, `status`=0";
		$this->Cluster_model->updateList($updatable);

		// 获取所有CDC服务器
		$sql = "SELECT * FROM cluster_cdc_server";
		$servers = $this->Cluster_model->getList($sql);

		// 记录离线的组
		$offline_groups = [];
		$leader_changes = [];

		foreach ($servers as $server) {
			// 测试节点连通性
			$host = trim($server['host_addr']);
			$port = trim($server['port']);
			$connection = @fsockopen($host, $port, $errno, $errstr, 1);

			if (!$connection) {
				// 节点离线
				$offline_groups[] = $server['group_name'];
				log_message('error', 'CDC节点离线: ' . $host . ':' . $port);
				continue;
			}

			fclose($connection);

			// 请求leader信息
			$url = 'http://' . $server['host_addr'] . ':' . $server['port'] . '/kunlun_cdc';
			$curlData = [
				"version" => "1.0",
				"job_id" => "",
				"job_type" => "get_leader",
				"timestamp" => time(),
				"user_name" => "kunlun_test"
			];

			$rs = $this->performCurlRequest($url, $curlData);
			if ($rs) {
				$resp = json_decode($rs, true);
				if ($resp['status'] == 'Done') {
					foreach ($resp['attachment'] as $job) {
						$masterArr = explode(":", $job);

						// Trim potential whitespace from host and port received from agent
						$current_leader_host = isset($masterArr[0]) ? trim($masterArr[0]) : '';
						$current_leader_port = isset($masterArr[1]) ? trim($masterArr[1]) : '';

						if (empty($current_leader_host) || empty($current_leader_port)) {
							log_message('error', 'Invalid leader format received from CDC agent for group ' . $server['group_name'] . ': ' . $job);
							continue; // Skip if format is invalid
						}

						// 更新主节点状态
						$updatable = "UPDATE cluster_cdc_server SET `master`='1', `status`='1' 
                                      WHERE `group_name`=? AND `host_addr`=? AND `port`=?";
						$this->Cluster_model->updateList($updatable, [
							$server['group_name'],
							$current_leader_host,
							$current_leader_port
						]);
						log_message('debug', 'Updated master status for group ' . $server['group_name'] . ' to ' . $current_leader_host . ':' . $current_leader_port);

						// Reconstruct the cleaned current leader string for comparison
						$current_leader = $current_leader_host . ':' . $current_leader_port;

						// 检测主备切换
						// Check if previous leader existed for this group
						if (isset($previous_leaders[$server['group_name']])) {
							$previous_leader_str = $previous_leaders[$server['group_name']];

							// Compare cleaned current leader with previous leader
							if ($previous_leader_str != $current_leader) {

								// Leader has changed, now check node count
								$group_nodes_query = "SELECT COUNT(*) as node_count FROM cluster_cdc_server WHERE `group_name`=?";
								$group_nodes_result = $this->Cluster_model->getList($group_nodes_query, [$server['group_name']]);
								$node_count = isset($group_nodes_result[0]['node_count']) ? intval($group_nodes_result[0]['node_count']) : 0;
								log_message('debug', 'Leader change detected for group ' . $server['group_name'] . '. Previous: [' . $previous_leader_str . '], Current: [' . $current_leader . ']. Node count: ' . $node_count);

								// 只有当节点数量大于1时才记录主备切换
								if ($node_count > 1) {
									// 记录切换信息
									$leader_changes[] = [
										'group_name' => $server['group_name'],
										'previous_leader' => $previous_leader_str,
										'current_leader' => $current_leader
									];
									log_message('info', 'Leader change added to list for group ' . $server['group_name'] . '. Previous: ' . $previous_leader_str . ', Current: ' . $current_leader);
								} else {
									// 只有一个节点时，不记录切换
									log_message('debug', 'CDC group ' . $server['group_name'] . ' has only ' . $node_count . ' node(s), leader change event ignored.');
								}
							} else {
								log_message('debug', 'Leader unchanged for group ' . $server['group_name'] . '. Leader: ' . $current_leader);
							}
						} else {
							// No previous leader known for this group
							log_message('info', 'No previous leader found for group ' . $server['group_name'] . '. Current leader set to: ' . $current_leader);
						}
					}
				}

				// 更新节点状态为在线
				$updatable = "UPDATE cluster_cdc_server SET `status`=1 
                              WHERE `group_name`=? AND `host_addr`=? AND `port`=?";
				$this->Cluster_model->updateList($updatable, [
					$server['group_name'],
					$server['host_addr'],
					$server['port']
				]);
			}
		}

		// 处理主备切换告警
		foreach ($leader_changes as $change) {
			$alarm_data = [
				'group_name' => $change['group_name'],
				'previous_leader' => $change['previous_leader'],
				'current_leader' => $change['current_leader'],
				'change_time' => date('Y-m-d H:i:s')
			];

			$this->recordAlarm('cdc_leader_change', '', $alarm_data, $alarm_data);

			log_message('warning', 'CDC主备切换: 组名=' . $change['group_name'] .
				', 旧主节点=' . $change['previous_leader'] .
				', 新主节点=' . $change['current_leader']);
		}

		log_message('debug', 'CDC集群leader节点获取和主备切换检测完成');
		return array_unique($offline_groups);
	}

	public function CdcListDumpJobs()
	{
		// 更新并检查CDC leader节点，同时获取离线节点组列表
		$offline_groups = $this->updateAndCheckCdcLeader();

		$master_sql = "SELECT * FROM cluster_cdc_server WHERE `master` = 1 GROUP BY group_name";
		$masterList = $this->Cluster_model->getList($master_sql);

		$all_groups_offline = empty($masterList);

		if ($all_groups_offline) {
			// 没有主节点的情况
			$this->handleNoMasterNodes();
			return;
		}

		// 如果所有组的主节点都离线，直接返回
		if (count($offline_groups) == count($masterList)) {
			$this->handleAllNodesOffline();
			return;
		}

		// 处理离线节点组的任务
		foreach ($offline_groups as $groupName) {
			// 获取组中主节点的信息，用于告警
			$node_sql = "SELECT host_addr, port FROM cluster_cdc_server WHERE group_name = ? LIMIT 1";
			$nodeInfo = $this->Cluster_model->getList($node_sql, [$groupName]);

			if (!empty($nodeInfo)) {
				$this->handleNodeOffline(
					$groupName,
					$nodeInfo[0]['host_addr'],
					$nodeInfo[0]['port']
				);
			}
		}

		// 记录离线的主节点组以便后续处理
		log_message('debug', 'CDC离线的组: ' . json_encode($offline_groups));

		if ($masterList) {
			// 同步CDC任务状态
			foreach ($masterList as $key => $value) {
				// 跳过离线组
				if (in_array($value['group_name'], $offline_groups)) {
					continue;
				}
				// 获取任务列表
				$this->syncCdcJobsList($masterList[$key]);
			}

			// 获取最新状态
			$sql = "SELECT a.*,b.* FROM cluster_cdc_worker a 
                    INNER JOIN cluster_cdc_server b ON a.group_name=b.group_name 
                    WHERE a.delete_time IS NULL GROUP BY a.job_id";
			$cdcWorkers = $this->Cluster_model->getList($sql);

			// 更新各任务状态详情
			$this->updateCdcWorkersDetail($cdcWorkers, $offline_groups);


			foreach ($cdcWorkers as $key => &$value) {
				$value['worker_param'] = json_decode($value['worker_param'], true);
			}

			$data['code'] = 200;
			$data['message'] = '获取ListDumps成功';
			$data['data'] = $cdcWorkers;
		} else {
			$data['code'] = 201;
			$data['message'] = '获取ListDumps成功';
			$data['data'] = [];
		}
		print_r(json_encode($data));
		return;
	}

	/**
	 * 处理所有节点都不存在的情况
	 */
	protected function handleNoMasterNodes()
	{
		log_message('error', 'CDC系统中没有找到主节点记录');

		// 获取所有未删除的CDC任务
		$all_jobs_sql = "SELECT job_id, group_name FROM cluster_cdc_worker WHERE delete_time IS NULL";
		$allJobsList = $this->Cluster_model->getList($all_jobs_sql);

		if ($allJobsList) {
			foreach ($allJobsList as $job) {
				// 更新任务状态为异常
				$this->updateCdcWorkerStatus(
					$job['job_id'],
					$job['group_name'],
					'no_master_node',
					urlencode('CDC系统中没有主节点记录'),
					false
				);

				// 记录告警信息
				$alarm_data = [
					'job_id' => $job['job_id'],
					'group_name' => $job['group_name'],
					'error_message' => 'CDC系统中没有主节点记录',
					'error_time' => date('Y-m-d H:i:s')
				];

				$this->recordAlarm('cdc_no_master_node', $job['job_id'], $alarm_data);
			}
		}

		$data['code'] = 201;
		$data['message'] = '所有CDC主节点离线';
		$data['data'] = [];
		print_r(json_encode($data));
	}

	/**
	 * 处理节点离线的情况
	 *
	 * @param string $groupName 组名称
	 * @param string $host 主机地址
	 * @param int $port 端口号
	 */
	protected function handleNodeOffline($groupName, $host, $port)
	{
		$jobs_sql = "SELECT job_id FROM cluster_cdc_worker WHERE group_name = ? AND delete_time IS NULL";
		$jobsList = $this->Cluster_model->getList($jobs_sql, [$groupName]);

		if ($jobsList) {
			foreach ($jobsList as $job) {
				// 更新任务状态为异常
				$this->updateCdcWorkerStatus(
					$job['job_id'],
					$groupName,
					'node_offline',
					urlencode('CDC主节点离线: ' . $host . ':' . $port),
					false
				);

				// 记录告警信息
				$alarm_data = [
					'job_id' => $job['job_id'],
					'group_name' => $groupName,
					'host_addr' => $host,
					'port' => $port,
					'error_message' => 'CDC主节点离线，无法连接',
					'error_time' => date('Y-m-d H:i:s')
				];

				$this->recordAlarm('cdc_node_offline', $job['job_id'], $alarm_data);
			}
		}
	}

	/**
	 * 处理所有节点都离线的情况
	 */
	protected function handleAllNodesOffline()
	{
		log_message('error', '所有CDC主节点都离线');

		// 获取所有未删除的CDC任务
		$all_jobs_sql = "SELECT job_id, group_name FROM cluster_cdc_worker WHERE delete_time IS NULL";
		$allJobsList = $this->Cluster_model->getList($all_jobs_sql);

		if ($allJobsList) {
			foreach ($allJobsList as $job) {
				// 更新任务状态为异常
				$this->updateCdcWorkerStatus(
					$job['job_id'],
					$job['group_name'],
					'all_nodes_offline',
					urlencode('所有CDC主节点都离线'),
					false
				);

				// 记录告警信息
				$alarm_data = [
					'job_id' => $job['job_id'],
					'group_name' => $job['group_name'],
					'error_message' => '所有CDC主节点都离线',
					'error_time' => date('Y-m-d H:i:s')
				];

				$this->recordAlarm('cdc_all_nodes_offline', $job['job_id'], $alarm_data);
			}
		}

		$data['code'] = 201;
		$data['message'] = '所有CDC主节点离线';
		$data['data'] = [];
		print_r(json_encode($data));
	}

	/**
	 * 同步CDC任务列表
	 *
	 * @param array $masterNode 主节点信息
	 */
	protected function syncCdcJobsList($masterNode)
	{
		$curlData = [
			"version" => "1.0",
			"job_id" => "",
			"job_type" => "list_dump_jobs",
			"timestamp" => time(),
			"user_name" => "kunlun_test"
		];

		$curl_addr = "http://" . trim($masterNode['host_addr']) . ":" . trim($masterNode['port']) . "/kunlun_cdc";
		$response = $this->GetCurlData($curl_addr, $curlData);

		if (!$response) {
			// 记录节点异常告警
			$alarm_data = [
				'group_name' => $masterNode['group_name'],
				'host_addr' => $masterNode['host_addr'],
				'port' => $masterNode['port'],
				'error_message' => 'CDC主节点无响应',
				'error_time' => date('Y-m-d H:i:s')
			];

			$this->recordAlarm('cdc_node_exception', '', $alarm_data, $alarm_data);

			log_message('error', 'CDC主节点异常: ' . $masterNode['host_addr'] . ':' . $masterNode['port'] .
				', 组名=' . $masterNode['group_name']);
			return;
		}

		// 处理响应数据
		$responseObj = json_decode($response, true);
		if (json_last_error() === JSON_ERROR_NONE && isset($responseObj['attachment']) && is_array($responseObj['attachment'])) {
			// 记录获取到的任务状态
			foreach ($responseObj['attachment'] as $item) {
				if (isset($item['job_id'])) {
					// 检查响应中是否包含超时错误
					$this->handleCdcConnectTimeout(
						$response,
						$item['job_id'],
						$masterNode['group_name'],
						[],
						$masterNode['host_addr'],
						$masterNode['port']
					);

					// 记录任务状态到cdc_job_state表
					// $this->saveJobState($item['job_id'], $masterNode['group_name'], $item);
				}
			}

			// 如果状态为Done，同步到数据库
			if ($responseObj['status'] == 'Done') {
				$this->syncJobsToDatabase($responseObj['attachment'], $masterNode['group_name']);
			}
		}
	}

	/**
	 * 将任务信息同步到数据库
	 *
	 * @param array $jobs 任务列表
	 * @param string $groupName 组名称
	 */
	protected function syncJobsToDatabase($jobs, $groupName)
	{
		// 获取该 groupName 下数据库中所有未删除的任务
		$existing_jobs_sql = "SELECT job_id FROM cluster_cdc_worker WHERE group_name = ? AND delete_time IS NULL";
		$existing_jobs = $this->Cluster_model->getList($existing_jobs_sql, [$groupName]);
		$existing_job_ids = [];

		if ($existing_jobs) {
			foreach ($existing_jobs as $job) {
				$existing_job_ids[] = $job['job_id'];
			}
			log_message('debug', '数据库中现有任务IDs: ' . implode(', ', $existing_job_ids));
		}

		// 记录当前 $jobs 中的所有 job_id
		$current_job_ids = [];

		// 处理 $jobs 中的每个任务
		foreach ($jobs as $job) {
			if (!isset($job['job_id']) || empty($job['job_id'])) {
				log_message('warning', 'CDC任务同步 - 跳过无效任务: job_id为空');
				continue;
			}

			// 添加到当前 job_ids 列表
			$current_job_ids[] = $job['job_id'];

			// 处理输出插件参数
			foreach ($job['output_plugins'] as $j => $js) {
				$job['output_plugins'][$j]['plugin_param'] = urlencode($js['plugin_param']);
			}

			// 查询是否已存在
			$sql = "SELECT * FROM cluster_cdc_worker WHERE job_id=? AND group_name=?";
			$params = [$job['job_id'], $groupName];
			$existingJob = $this->Cluster_model->getList($sql, $params);

			if (!$existingJob) {
				// 插入新任务
				$sql = "INSERT INTO cluster_cdc_worker (`worker_param`, `create_time`, `status`, `group_name`, `job_id`) VALUES (?, ?, ?, ?, ?)";
				$params = [
					json_encode($job),
					time(),
					'running',
					$groupName,
					$job['job_id']
				];
				$this->Cluster_model->updateList($sql, $params);
				log_message('debug', '创建新CDC任务记录: job_id=' . $job['job_id']);
			} else {
				// 更新现有任务，将其标记为活动状态
				$update = "UPDATE cluster_cdc_worker SET `worker_param`=?, `status`='running', `delete_time`=NULL WHERE job_id=? AND group_name=?";
				$params = [
					json_encode($job),
					$job['job_id'],
					$groupName
				];
				$this->Cluster_model->updateList($update, $params);
				log_message('debug', '更新现有CDC任务记录: job_id=' . $job['job_id']);
			}
		}

		// 找出数据库中存在但 $jobs 中不存在的记录，将其状态设置为 deleted
		if (!empty($existing_job_ids) && !empty($current_job_ids)) {
			$deleted_job_ids = array_diff($existing_job_ids, $current_job_ids);

			if (!empty($deleted_job_ids)) {
				log_message('debug', '需要标记为已删除的任务IDs: ' . implode(', ', $deleted_job_ids));

				foreach ($deleted_job_ids as $deleted_job_id) {
					$update_deleted = "UPDATE cluster_cdc_worker SET `status`='异常', `worker_status`='异常' WHERE job_id=? AND group_name=?";
					$params = [$deleted_job_id, $groupName];
					$this->Cluster_model->updateList($update_deleted, $params);

					log_message('debug', '标记任务为已删除: job_id=' . $deleted_job_id . ', group_name=' . $groupName);
				}
			}
		}
	}

	/**
	 * 更新CDC任务详情
	 *
	 * @param array $cdcWorkers 待更新的任务列表
	 * @param array $offlineGroups 离线组列表
	 */
	protected function updateCdcWorkersDetail($cdcWorkers, $offlineGroups)
	{
		foreach ($cdcWorkers as $key => $value) {
			// 跳过离线节点组的任务
			if (in_array($value['group_name'], $offlineGroups)) {
				continue;
			}

			$job = json_decode($value['worker_param'], true);
			if (!is_array($job)) {
				continue;
			}

			// 解码输出插件参数
			foreach ($job['output_plugins'] as $k => $jobs) {
				$job['output_plugins'][$k]['plugin_param'] = urldecode($jobs['plugin_param']);
			}

			$cdcWorkers[$key]['worker_param'] = $job;

			// 获取任务状态
			$curlData = [
				"version" => "1.0",
				"job_id" => "",
				"job_type" => "get_job_state",
				"timestamp" => time(),
				"user_name" => "kunlun_test",
				"paras" => [
					"meta_db" => $job['metadb'],
					"cluster_name" => $job['cluster_name'],
					"dump_tables" => $job['dump_tables']
				]
			];

			$curl_addr = "http://" . trim($cdcWorkers[$key]['host_addr']) . ":" . trim($cdcWorkers[$key]['port']) . "/kunlun_cdc";
			$responseItem = $this->GetCurlData($curl_addr, $curlData);

			if ($responseItem) {
				// 检查是否存在超时错误
				$timeoutDetected = $this->handleCdcConnectTimeout(
					$responseItem,
					$job['job_id'],
					$value['group_name'],
					$job,
					$cdcWorkers[$key]['host_addr'],
					$cdcWorkers[$key]['port']
				);

				if ($timeoutDetected) {
					continue;
				}

				// 处理响应数据
				$responseItemArr = json_decode($responseItem, true);

//				pa($responseItemArr);
				$cdcWorkers[$key]['states'] = $responseItemArr;

				// 处理任务状态
				$this->processJobStates($responseItemArr, $job, $value, $cdcWorkers[$key]);
			} else {

				// 节点无响应
				$this->handleNodeNoResponse($job, $value, $cdcWorkers[$key]);
				$cdcWorkers[$key]['states'] = null;
			}
		}
	}

	/**
	 * 处理任务状态
	 *
	 * @param array $response 响应数据
	 * @param array $job 任务数据
	 * @param array $workerInfo 工作节点信息
	 * @param array $serverInfo 服务器信息
	 */
	protected function processJobStates($response, $job, $workerInfo, $serverInfo)
	{
		if (empty($response['attachment'])) {

			// 空响应时使用默认状态
			$this->updateCdcWorkerStatus(
				$job['job_id'],
				$workerInfo['group_name'],
				'',
				'',
				true
			);
			return;
		}

		foreach ($response['attachment'] as $kv) {
			if (!isset($kv['sync_state']) || empty($kv['sync_state'])) {
				continue;
			}

			// 解析状态数据
			$status_info = json_decode($kv['sync_state'], true);
			if (json_last_error() !== JSON_ERROR_NONE) {
				log_message('error', '解析sync_state失败：' . json_last_error_msg());
				continue;
			}

			//print_r($status_info);

			// 获取历史状态记录
			$cdc_job_state_sql = "SELECT * FROM cdc_job_state WHERE job_id=? AND group_name=? ORDER BY id DESC LIMIT 1";
			$params_job = [
				$job['job_id'],
				$workerInfo['group_name']
			];
			$get_job_state = $this->Cluster_model->getList($cdc_job_state_sql, $params_job);


			// 检查是否有错误状态
			$hasErrors = $this->checkAndProcessErrors($status_info, $job, $workerInfo, $serverInfo);
			if ($hasErrors) {
				continue;
			}

			// 检查是否需要创建新记录
			$shouldInsertNewRecord = $this->shouldCreateNewStateRecord($status_info, $get_job_state);

			if ($shouldInsertNewRecord) {
				// 创建新记录
				$jobsql = "INSERT INTO `cdc_job_state` (`param`, `job_id`, `group_name`, `time`) VALUES (?, ?, ?, ?)";
				$params = [
					$kv['sync_state'],
					$job['job_id'],
					$workerInfo['group_name'],
					time()
				];
				$this->Cluster_model->updateList($jobsql, $params);
			} else if (!empty($get_job_state)) {
				$jobsql = "INSERT INTO `cdc_job_state` (`param`, `job_id`, `group_name`, `time`) VALUES (?, ?, ?, ?)";
				$params = [
					$kv['sync_state'],
					$job['job_id'],
					$workerInfo['group_name'],
					time()
				];
				$this->Cluster_model->updateList($jobsql, $params);
			}

			// 更新Worker服务状况
			$dumpState = isset($kv['dump_state']) ? $kv['dump_state'] : '';
			$this->updateCdcWorkerStatus(
				$job['job_id'],
				$workerInfo['group_name'],
				$dumpState,
				""
			);

			// 执行定期清理
			$this->periodicStateCleanup($job['job_id']);
		}

		// 检查CDC解析DDL错误
		$this->checkDdlParseErrors($response['attachment'], $job, $workerInfo);
	}

	/**
	 * 检查和处理错误状态
	 *
	 * @param array $status_info 状态信息
	 * @param array $job 任务数据
	 * @param array $workerInfo 工作节点信息
	 * @param array $serverInfo 服务器信息
	 * @return bool 是否存在错误
	 */
	protected function checkAndProcessErrors($status_info, $job, $workerInfo, $serverInfo)
	{
		if (!isset($status_info['dispatch_event_state']) || !is_array($status_info['dispatch_event_state'])) {
			return false;
		}

		$hasErrors = false;

		foreach ($status_info['dispatch_event_state'] as $item_status) {
			if (!isset($item_status['channel_name'])) {
				continue;
			}

			if ($item_status['channel_name'] == 'event_file') {
				continue;
			}

			if (!isset($item_status['dispatch_result']) || $item_status['dispatch_result'] != 'ok') {
				$errorMsg = isset($item_status['dispatch_result']) ? $item_status['dispatch_result'] : '未知错误';

				$this->updateCdcWorkerStatus(
					$job['job_id'],
					$workerInfo['group_name'],
					"异常",
					urlencode($errorMsg),
					false
				);

				// 记录同步目标数据库失败告警
				$alarm_data = [
					'job_id' => $job['job_id'],
					'group_name' => $workerInfo['group_name'],
					'error_message' => $errorMsg,
					'dispatch_result' => $errorMsg,
					'channel_name' => $item_status['channel_name'],
					'cluster_name' => $job['cluster_name'],
					'metadb' => $job['metadb'],
					'error_time' => date('Y-m-d H:i:s'),
					'sync_state' => $status_info,
					'tables' => $job['dump_tables']
				];

				$this->recordAlarm('cdc_sync_target_failed', $job['job_id'], $alarm_data);

				$hasErrors = true;
				break;
			}
		}

		return $hasErrors;
	}

	/**
	 * 检查是否应该创建新的状态记录
	 *
	 * @param array $status_info 状态信息
	 * @param array $get_job_state 历史状态记录
	 * @return bool 是否应该创建新记录
	 */
	protected function shouldCreateNewStateRecord($status_info, $get_job_state)
	{


		if (empty($get_job_state)) {
			return true;
		}

		$state = json_decode($get_job_state[0]['param'], true);
		if (json_last_error() !== JSON_ERROR_NONE) {
			return true;
		}

		//pae([$status_info, $get_job_state, $state]);

		// 检查关键指标变化
		if (isset($state['sync_sql_num']) && isset($status_info['sync_sql_num']) &&
			$state['sync_sql_num'] != $status_info['sync_sql_num'] && $status_info['sync_sql_num'] != 0) {
			return true;
		}

		// 检查缓存和待处理SQL数量变化
		$significantChange = false;
		if (isset($state['cache_sql_num']) && isset($status_info['cache_sql_num']) &&
			$state['cache_sql_num'] != $status_info['cache_sql_num']) {
			$significantChange = true;
		}

		if (isset($state['pending_sql_num']) && isset($status_info['pending_sql_num']) &&
			$state['pending_sql_num'] != $status_info['pending_sql_num']) {
			$significantChange = true;
		}

		// 检查错误状态变化
		$previousHasError = $this->hasErrorStatus($state);
		$currentHasError = $this->hasErrorStatus($status_info);

		if ($previousHasError != $currentHasError) {
			return true;
		}

		// 检查时间间隔
		$lastRecordTime = isset($get_job_state[0]['time']) ? (int)$get_job_state[0]['time'] : 0;
		$currentTime = time();
		$timeInterval = 3600; // 1小时

		if (($currentTime - $lastRecordTime) >= $timeInterval) {
			return true;
		}

		return $significantChange;
	}

	/**
	 * 检查状态是否包含错误
	 *
	 * @param array $state 状态信息
	 * @return bool 是否包含错误
	 */
	protected function hasErrorStatus($state)
	{
		if (!isset($state['dispatch_event_state']) || !is_array($state['dispatch_event_state'])) {
			return false;
		}

		foreach ($state['dispatch_event_state'] as $event) {
			if (isset($event['dispatch_result']) && $event['dispatch_result'] != 'ok') {
				return true;
			}
		}

		return false;
	}

	/**
	 * 定期清理过时状态数据
	 *
	 * @param string $jobId 任务ID
	 */
	protected function periodicStateCleanup($jobId)
	{
		// 随机执行清理操作，减少频率
		if (rand(1, 100) <= 5) {
			// 保留最近30天的数据，每个job_id最多保留100条记录
			$cleanupTime = time() - (86400 * 30); // 30天前

			$cleanup_sql = "DELETE FROM cdc_job_state WHERE time < ? AND job_id = ? AND id NOT IN (
                SELECT id FROM (
                    SELECT id FROM cdc_job_state 
                    WHERE job_id = ? 
                    ORDER BY time DESC 
                    LIMIT 100
                ) AS temp
            )";

			$this->Cluster_model->updateList($cleanup_sql, [
				$cleanupTime,
				$jobId,
				$jobId
			]);
		}
	}

	/**
	 * 检查CDC解析DDL错误
	 *
	 * @param array $attachments 附件数据
	 * @param array $job 任务数据
	 * @param array $workerInfo 工作节点信息
	 */
	protected function checkDdlParseErrors($attachments, $job, $workerInfo)
	{
		foreach ($attachments as $kv) {
			if (isset($kv['dump_state']) && $kv['dump_state'] == 'stop by cdc inner error') {
				// 记录CDC解析DDL失败告警
				$alarm_data = [
					'job_id' => $job['job_id'],
					'group_name' => $workerInfo['group_name'],
					'error_message' => 'stop by cdc inner error',
					'dump_state' => $kv['dump_state'],
					'cluster_name' => $job['cluster_name'],
					'metadb' => $job['metadb'],
					'error_time' => date('Y-m-d H:i:s'),
					'sync_state' => isset($kv['sync_state']) ? json_decode($kv['sync_state'], true) : null,
					'tables' => $job['dump_tables']
				];

				$this->recordAlarm('cdc_parse_ddl_failed', $job['job_id'], $alarm_data);

				// 更新CDC任务状态
				$this->updateCdcWorkerStatus(
					$job['job_id'],
					$workerInfo['group_name'],
					'异常',
					urlencode('CDC解析DDL失败: ' . $kv['dump_state']),
					false
				);
			}
		}
	}

	/**
	 * 处理节点无响应的情况
	 *
	 * @param array $job 任务数据
	 * @param array $workerInfo 工作节点信息
	 * @param array $serverInfo 服务器信息
	 */
	protected function handleNodeNoResponse($job, $workerInfo, $serverInfo)
	{
		// 记录CDC节点异常告警
		$alarm_data = [
			'job_id' => $job['job_id'],
			'group_name' => $workerInfo['group_name'],
			'host_addr' => $serverInfo['host_addr'],
			'port' => $serverInfo['port'],
			'error_message' => 'CDC节点不响应API请求',
			'cluster_name' => $job['cluster_name'],
			'metadb' => $job['metadb'],
			'error_time' => date('Y-m-d H:i:s')
		];

		$this->recordAlarm('cdc_node_exception', $job['job_id'], $alarm_data);

		// 更新状态
		$this->updateCdcWorkerStatus(
			$job['job_id'],
			$workerInfo['group_name'],
			'',
			'节点不响应API请求',
			true
		);
	}

	/**
	 * 记录任务状态
	 *
	 * @param string $jobId 任务ID
	 * @param string $groupName 组名称
	 * @param array $jobData 任务数据
	 */
	protected function saveJobState($jobId, $groupName, $jobData)
	{
		$jobsql = "INSERT INTO `cdc_job_state` (`param`, `job_id`, `group_name`, `time`) VALUES (?,?,?,?)";
		$params = [
			json_encode($jobData),
			$jobId,
			$groupName,
			time()
		];
		$this->Cluster_model->updateList($jobsql, $params);
	}

	/**
	 * 记录告警信息
	 *
	 * @param string $alarmType 告警类型
	 * @param string $jobId 任务ID
	 * @param array $alarmData 告警数据
	 * @param array $compareData 用于比较的数据，默认与告警数据相同
	 */
	protected function recordAlarm($alarmType, $jobId, $alarmData, $compareData = null)
	{
		if ($compareData === null) {
			$compareData = $alarmData;
		}

		$msgData = json_encode($alarmData);
		$compareJson = json_encode($compareData);

		// 检查是否已存在未处理的告警
		if (!empty($jobId)) {
			$select_sql = "SELECT id FROM cluster_alarm_info WHERE alarm_type=? AND job_id=? AND status='unhandled'";
			$res_select = $this->Cluster_model->getList($select_sql, [$alarmType, $jobId]);
		} else {
			$select_sql = "SELECT id FROM cluster_alarm_info WHERE alarm_type=? AND job_info=? AND status='unhandled'";
			$res_select = $this->Cluster_model->getList($select_sql, [$alarmType, $compareJson]);
		}

		if (empty($res_select)) {
			// 插入新告警
			if (!empty($jobId)) {
				$insert_sql = "INSERT INTO cluster_alarm_info(alarm_type, job_info, occur_timestamp, job_id) VALUES (?, ?, NOW(), ?)";
				$this->Cluster_model->updateList($insert_sql, [$alarmType, $msgData, $jobId]);
			} else {
				$insert_sql = "INSERT INTO cluster_alarm_info(alarm_type, job_info, occur_timestamp) VALUES (?, ?, NOW())";
				$this->Cluster_model->updateList($insert_sql, [$alarmType, $msgData]);
			}
		} else {
			// 更新现有告警
			$update_sql = "UPDATE cluster_alarm_info SET job_info=? WHERE id=?";
			$this->Cluster_model->updateList($update_sql, [$msgData, $res_select[0]['id']]);
		}
	}

	public function cdcEditWork()
	{
		// 开始记录请求信息
		log_message('debug', '===cdcEditWork - 开始处理请求');

		//获取token
		$arr = apache_request_headers();
		$token = $arr["Token"];
		if (empty($token)) {
			log_message('error', '===cdcEditWork - Token为空');
			$data['code'] = 201;
			$data['message'] = 'token不能为空';
			print_r(json_encode($data));
			return;
		}

		$this->load->model('Cluster_model');
		$this->load->model('Cdc_model');

		//判断参数
		$string = json_decode(@file_get_contents('php://input'), true);
		log_message('debug', '===cdcEditWork - 接收到的原始数据: ' . json_encode($string));

		$stream_id = $string['stream_id'];
		$stream_type = $string['stream_type'];
		$hot_switch = $string['hot_switch'];

		$job_name = $string['job_name'];
		log_message('debug', '===cdcEditWork - job_name值: ' . (isset($job_name) ? $job_name : 'undefined'));

		if ($job_name == "") {
			log_message('error', '===cdcEditWork - job_name为空');
			$data['code'] = 201;
			$data['message'] = 'job_name不能为空';
			print_r(json_encode($data));
			return;
		}

		$src_db_type = $string['src_db_type'];
		if ($src_db_type == "") {
			log_message('error', '===cdcEditWork - src_db_type为空');
			$data['code'] = 201;
			$data['message'] = 'src_db_type不能为空';
			print_r(json_encode($data));
			return;
		}

		$dst_db_type = $string['dst_db_type'];
		if ($dst_db_type == "") {
			log_message('error', '===cdcEditWork - dst_db_type为空');
			$data['code'] = 201;
			$data['message'] = 'dst_db_type不能为空';
			print_r(json_encode($data));
			return;
		}

		$db_cluster_id = $string['db_cluster_id'];
		if ($db_cluster_id == "") {
			log_message('error', '===cdcEditWork - db_cluster_id为空');
			$data['code'] = 201;
			$data['message'] = 'db_cluster_id不能为空';
			print_r(json_encode($data));
			return;
		}

		$is_proxy_sql = $string['is_proxy_sql'];
		if ($is_proxy_sql == "") {
			log_message('error', '===cdcEditWork - is_proxy_sql为空');
			$data['code'] = 201;
			$data['message'] = 'is_proxy_sql不能为空';
			print_r(json_encode($data));
			return;
		}

		$group_name = $string['paras']['cdc_group_name'];
		log_message('debug', '===cdcEditWork - group_name值: ' . $group_name);

		//获取cdc 主
		$master_sql = "select * from cluster_cdc_server where master=1 and group_name=?";
		$parms = [
			$string['paras']['cdc_group_name']
		];
		log_message('debug', '===cdcEditWork - 执行master查询SQL: ' . $master_sql . ', 参数: ' . json_encode($parms));

		$masterList = $this->Cluster_model->getList($master_sql, $parms);
		log_message('debug', '===cdcEditWork - master查询结果: ' . json_encode($masterList));

		unset($string['paras']['cdc_group_name']);
		unset($string['stream_type']);
		unset($string['stream_id']);
		unset($string['hot_switch']);
		unset($string['job_name']);
		unset($string['src_db_type']);
		unset($string['dst_db_type']);
		unset($string['db_cluster_id']);
		unset($string['is_proxy_sql']);

		$dump_db_type = $string['paras']['dump_db_type'];
		$output_plugins = $string['paras']['output_plugins'];
		if ($dump_db_type == 'mongodb' && $output_plugins[0]['plugin_name'] === 'event_sql') {

			$database_name = $string['paras']['shard_params'][0]['binlog_file'];
			$table_name = $string['paras']['shard_params'][0]['binlog_pos'];
			$plugin_param = json_decode($output_plugins[0]['plugin_param'], true);
			$kunlun_cdc_dump_state_sql = "CREATE TABLE IF NOT EXISTS kunlun_cdc_dump_state (`job_id` varchar(64) not NULL,`repl_info` varchar(1024) default '',PRIMARY KEY(`job_id`))";
			$this->Cdc_model->getMysqlNullDb($plugin_param['hostaddr'], $plugin_param['port'], $plugin_param['user'], $plugin_param['password'], "public", $kunlun_cdc_dump_state_sql);

			$create_table_sql = "CREATE TABLE IF NOT EXISTS ${table_name} (`__sys_obj_id__` VARCHAR PRIMARY KEY, `doc` LONGTEXT,`optype` VARCHAR DEFAULT 'insert')";
			$this->Cdc_model->getMysqlNullDb($plugin_param['hostaddr'], $plugin_param['port'], $plugin_param['user'], $plugin_param['password'], $database_name, $create_table_sql);
		}


		if ($dump_db_type == 'tdengine' && $output_plugins[0]['plugin_name'] === 'event_sql') {

			$database_name = $string['paras']['shard_params'][0]['binlog_file'];
			$table_name = $string['paras']['shard_params'][0]['binlog_pos'];
			$plugin_param = json_decode($output_plugins[0]['plugin_param'], true);
			$kunlun_cdc_dump_state_sql = "CREATE TABLE IF NOT EXISTS kunlun_cdc_dump_state (`job_id` varchar(64) not NULL,`repl_info` varchar(1024) default '',PRIMARY KEY(`job_id`))";
			$this->Cdc_model->getMysqlNullDb($plugin_param['hostaddr'], $plugin_param['port'], $plugin_param['user'], $plugin_param['password'], "public", $kunlun_cdc_dump_state_sql);

			$create_table_sql = "CREATE TABLE IF NOT EXISTS ${table_name} (`id` VARCHAR PRIMARY KEY, `name` VARCHAR,`ts` VARCHAR)";
			$this->Cdc_model->getMysqlNullDb($plugin_param['hostaddr'], $plugin_param['port'], $plugin_param['user'], $plugin_param['password'], $database_name, $create_table_sql);
		}

		// if ($output_plugins[0]['plugin_name'] === 'event_mongodb') {

		//     //需要清空存在的记录
		//     $database_name = $string['paras']['dump_tables'];
		//     $database_name_arr = explode("_", $database_name);
		//     $table_item = explode(".", $database_name_arr[2]);
		//     $databses = $table_item[0];
		//     $collection = $table_item[1];
		//     $plugin_param = json_decode($output_plugins[0]['plugin_param'], true);
		//     $mongo_client = $plugin_param['mongo_url'];

		//     // 使用已加载的 MongoDB 模型
		//     $this->MongoDB_model->del_db_list($mongo_client, $databses, $collection);
		// }

		$resp = $this->Cdc_model->add_cdc_worker_list($group_name, $string);

		if ($hot_switch == "1") {

			$insert_sql = "INSERT INTO `cluster_cdc_hot_switch` (`cdc_job_id`) VALUES (?)";
			$params = [
				$resp['job_id']
			];
			$this->Cluster_model->updateList($insert_sql, $params);
		}

		if ($resp['code'] == 200) {
			$insert_sql = "INSERT INTO cluster_cdc_worker(create_time,group_name,job_name,src_db_type,dst_db_type,db_cluster_id,is_proxy_sql,job_id) VALUES (?,?,?,?,?,?,?,?)";

			$params = [
				time(),
				$group_name,
				$job_name,
				$src_db_type,
				$dst_db_type,
				$db_cluster_id,
				$is_proxy_sql,
				$resp['job_id']
			];

			log_message('debug', '===cdcEditWork - 执行插入SQL: ' . $insert_sql);
			log_message('debug', '===cdcEditWork - 插入参数: ' . json_encode($params));

			$insert_result = $this->Cluster_model->updateList($insert_sql, $params);
			log_message('debug', '===cdcEditWork - 插入结果: ' . ($insert_result ? 'success' : 'failed'));

			$sql = "update cluster_steams set ";
			if ($stream_type == 0) {
				$sql .= "cdc_src=?";
			} else {
				$sql .= "cdc_dst=?";
			}
			$sql .= " where task_id=?";

			$stream_parms = [
				$resp['job_id'],
				$stream_id
			];

			log_message('debug', '===cdcEditWork - 执行stream更新SQL: ' . $sql);
			log_message('debug', '===cdcEditWork - 更新参数: ' . json_encode($stream_parms));

			$update_result = $this->Cluster_model->updateList($sql, $stream_parms);
			log_message('debug', '===cdcEditWork - 更新结果: ' . ($update_result ? 'success' : 'failed'));
		}

		log_message('debug', '===cdcEditWork - 处理完成，返回结果: ' . json_encode($resp));
		print_r(json_encode($resp));
	}

	/**
	 * 检测和处理CDC连接超时错误
	 *
	 * @param string $responseData CURL返回的响应数据
	 * @param string $jobId 相关的任务ID
	 * @param string $groupName CDC组名称
	 * @param array $jobInfo 任务详细信息
	 * @param string $hostAddr 主机地址
	 * @param int $port 端口号
	 * @return bool 是否检测到并处理了超时错误
	 */
	protected function handleCdcConnectTimeout($responseData, $jobId, $groupName, $jobInfo = [], $hostAddr = '', $port = 0)
	{
		// 检查是否是JSON格式
		$responseArr = json_decode($responseData, true);
		if (json_last_error() !== JSON_ERROR_NONE) {
			log_message('debug', '===handleCdcConnectTimeout - 响应不是有效JSON: ' . json_last_error_msg());
			return false;
		}

		log_message('debug', '===handleCdcConnectTimeout - 开始检查超时错误，jobId: ' . $jobId . ', groupName: ' . $groupName);

		// 检查超时错误模式1: 响应中直接包含超时信息
		if (
			strpos($responseData, 'Max connect timeout') !== false ||
			strpos($responseData, 'connect timeout') !== false ||
			strpos($responseData, 'timeout reached') !== false
		) {

			log_message('debug', '===handleCdcConnectTimeout - 检测到超时关键词');

			// 提取错误详情
			$errorMessage = '';
			if (isset($responseArr['attachment']) && is_array($responseArr['attachment'])) {
				foreach ($responseArr['attachment'] as $item) {
					if (isset($item['sync_state'])) {
						$syncState = json_decode($item['sync_state'], true);
						if (json_last_error() === JSON_ERROR_NONE && isset($syncState['dispatch_event_state'])) {
							foreach ($syncState['dispatch_event_state'] as $event) {
								if (
									isset($event['dispatch_result']) &&
									(strpos($event['dispatch_result'], 'timeout') !== false ||
										strpos($event['dispatch_result'], 'Max connect timeout') !== false)
								) {
									$errorMessage = $event['dispatch_result'];
									log_message('debug', '===handleCdcConnectTimeout - 找到超时错误详情: ' . $errorMessage);
									break;
								}
							}
						}
					}
				}
			}

			if (empty($errorMessage)) {
				$errorMessage = '数据库连接超时，请检查CDC与目标数据库的连接';
			}

			// 构建告警数据
			$alarmData = [
				'job_id' => $jobId,
				'group_name' => $groupName,
				'error_message' => $errorMessage,
				'error_time' => date('Y-m-d H:i:s'),
				'host_addr' => $hostAddr,
				'port' => $port
			];

			// 如果有额外信息，添加到告警中
			if (!empty($jobInfo)) {
				$alarmData = array_merge($alarmData, [
					'cluster_name' => isset($jobInfo['cluster_name']) ? $jobInfo['cluster_name'] : '',
					'metadb' => isset($jobInfo['metadb']) ? $jobInfo['metadb'] : '',
					'tables' => isset($jobInfo['dump_tables']) ? $jobInfo['dump_tables'] : ''
				]);
			}

			$msgData = json_encode($alarmData);
			log_message('debug', '===handleCdcConnectTimeout - 构建的告警数据: ' . substr($msgData, 0, 100) . '...');

			// 检查是否已存在未处理的告警
			$selectSql = "SELECT id FROM cluster_alarm_info WHERE alarm_type='cdc_db_connect_timeout' AND job_id=? AND status='unhandled'";
			log_message('debug', '===handleCdcConnectTimeout - 查询是否存在告警: job_id=' . $jobId);

			$resSelect = $this->Cluster_model->getList($selectSql, [$jobId]);

			if (empty($resSelect)) {
				// 记录新告警
				$insertSql = "INSERT INTO cluster_alarm_info(alarm_type,job_info,occur_timestamp,job_id) VALUES (?,?,NOW(),?)";
				$insertResult = $this->Cluster_model->updateList($insertSql, ['cdc_db_connect_timeout', $msgData, $jobId]);
				log_message('debug', '===handleCdcConnectTimeout - 插入超时告警结果: ' . ($insertResult ? '成功' : '失败'));

				log_message('error', 'CDC连接目标数据库超时: Job ID: ' . $jobId . ', Error: ' . $errorMessage);
			} else {
				// 更新现有告警
				$updateSql = "UPDATE cluster_alarm_info SET job_info=? WHERE id=?";
				$updateResult = $this->Cluster_model->updateList($updateSql, [$msgData, $resSelect[0]['id']]);
				log_message('debug', '===handleCdcConnectTimeout - 更新超时告警结果: ' . ($updateResult ? '成功' : '失败'));
			}

			// 更新worker状态
			$this->updateCdcWorkerStatus(
				$jobId,
				$groupName,
				"异常",
				urlencode($errorMessage),
				false
			);

			return true;
		}

		return false;
	}

	/**
	 * 更新CDC Worker的状态，确保状态正确反映dump_state
	 *
	 * @param string $jobId 任务ID
	 * @param string $groupName 组名
	 * @param string $dumpState 当前的dump_state值
	 * @param string $errorMessage 错误信息，如果有的话
	 * @param bool $useHistoricalState 是否在当前状态无效时尝试使用历史状态
	 * @return void
	 */
	protected function updateCdcWorkerStatus($jobId, $groupName, $dumpState = '', $errorMessage = '', $useHistoricalState = true)
	{
		// 如果提供了有效的dump_state，直接使用
		if (!empty($dumpState)) {
			// 当 dumpState 为"异常"时，同时将 status 设置为"异常"
			$status = ($dumpState === '异常') ? '异常' : 'running';

			$update = "UPDATE cluster_cdc_worker SET status=?, worker_status=?, error_message=? WHERE job_id=? AND group_name=?";
			$params = [
				$status,
				$dumpState,
				$errorMessage,
				$jobId,
				$groupName
			];
			$this->Cluster_model->updateList($update, $params);
			log_message('debug', 'CDC状态更新：使用当前dump_state=' . $dumpState .
				'，状态=' . $status .
				'，作业ID=' . $jobId . '，组名=' . $groupName .
				(empty($errorMessage) ? '' : '，错误=' . $errorMessage));
			return;
		}

		// 如果没有提供有效的dump_state且允许使用历史状态，尝试从历史记录获取
		if ($useHistoricalState) {
			$jobStateQuery = "SELECT param FROM cdc_job_state WHERE job_id = ? AND group_name = ? ORDER BY time DESC LIMIT 1";
			$lastJobState = $this->Cluster_model->getList($jobStateQuery, [$jobId, $groupName]);

			if (!empty($lastJobState) && isset($lastJobState[0]['param'])) {
				$lastState = json_decode($lastJobState[0]['param'], true);
				if (json_last_error() === JSON_ERROR_NONE && isset($lastState['dump_state']) && !empty($lastState['dump_state'])) {
					// 使用最后一次记录的dump_state
					$historyDumpState = $lastState['dump_state'];
					// 当 historyDumpState 为"异常"时，同时将 status 设置为"异常"
					$status = ($historyDumpState === '异常') ? '异常' : 'running';

					$update = "UPDATE cluster_cdc_worker SET status=?, worker_status=?, error_message=? WHERE job_id=? AND group_name=?";
					$params = [
						$status,
						$historyDumpState,
						$errorMessage,
						$jobId,
						$groupName
					];
					$this->Cluster_model->updateList($update, $params);
					log_message('debug', 'CDC状态更新：使用历史记录的dump_state=' . $historyDumpState .
						'，状态=' . $status .
						'，作业ID=' . $jobId . '，组名=' . $groupName .
						(empty($errorMessage) ? '' : '，错误=' . $errorMessage));
					return;
				}
			}
		}

		// 如果无法获取有效状态，使用默认状态
		$defaultStatus = empty($errorMessage) ? 'unknown' : 'error';
		$update = "UPDATE cluster_cdc_worker SET status=?, worker_status=?, error_message=? WHERE job_id=? AND group_name=?";
		$params = [
			$defaultStatus,
			$defaultStatus,
			$errorMessage,
			$jobId,
			$groupName
		];
		$this->Cluster_model->updateList($update, $params);
		log_message('debug', 'CDC状态更新：使用默认状态=' . $defaultStatus .
			'，作业ID=' . $jobId . '，组名=' . $groupName .
			(empty($errorMessage) ? '' : '，错误=' . $errorMessage));
	}
}
