<?php
defined('BASEPATH') or exit('No direct script access allowed');

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// 引入 Composer 的自动加载文件
require_once FCPATH . 'vendor/autoload.php';

class Monitor extends CI_Controller
{
    public $cache_duration = 300; // 缓存时间 5 分钟
    public $alarm_config = null;
    public $push_config = null;
    public $config;
    public $key;
    public $post_url;
    public $default_username;
    public $pg_username;
    public $db_prefix;
    public $Login_model;
    public $Cluster_model;
    
    // 添加新的类库引用
    public $alarmhandler;
    public $notificationhandler;
    public $performancemonitor;

    public function __construct()
    {
        parent::__construct();
        //打开重定向
        header('Access-Control-Allow-Origin:*'); // *代表允许任何网址请求
        header('Access-Control-Allow-Headers: Content-Type,Content-Length, Origin'); // 设置允许自定义请求头的字段
        header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE'); // 允许请求的类型
        header('Access-Control-Allow-Headers:x-requested-with,content-type,Token'); //允许接受token
        header('Content-Type: application/json;charset=utf-8');

        // 加载必要的模型和库
        $this->load->model('Login_model');
        $this->load->model('Cluster_model');
        
        // 加载自定义类库
        $this->load->library('AlarmHandler');
        $this->load->library('NotificationHandler');
        $this->load->library('PerformanceMonitor');
        
        // 加载表单辅助函数
        $this->load->helper('form');
        
        // 加载配置
        $this->config->load('myconfig');
        
        // 初始化配置
        $monitor_config = $this->config->item('monitor');
        $this->key = $monitor_config['key'] ?? '';
    }

    private function initAlarmConfig()
    {
        // 从缓存获取告警配置
        $sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_message_config";
        $this->alarm_config = $this->Cluster_model->getList($sql);

        // 从缓存获取推送配置
        $sql = "SELECT * FROM cluster_alarm_message_config";
        $this->push_config = $this->Cluster_model->getList($sql);
    }

    public function getClusterMonitor()
    {
        set_time_limit(0);
        log_message('debug', '=== getClusterMonitor() Start ===');
        try {
            // 获取并验证用户
            $serve = $_SERVER['QUERY_STRING'];
            $string = preg_split('/[=&]/', $serve);
            $arr = array();
            for ($i = 0; $i < count($string); $i += 2) {
                if (isset($string[$i+1])) {
                    $arr[$string[$i]] = $string[$i + 1];
                }
            }
            $user_name = $arr['user_name'] ?? '';
            
            if (empty($user_name)) {
                throw new Exception('用户名不能为空');
            }

            $user_sql = "select id from kunlun_dba_tools_db.kunlun_user where name=?";
            $res_user = $this->Login_model->getList($user_sql, [$user_name]);
            if (empty($res_user)) {
                log_message('error', 'User validation failed: user does not exist');
                throw new Exception('用户不存在');
            }
            $user_id = $res_user[0]['id'];
            log_message('debug', 'User validated successfully: ' . $user_name);

            // 获取所有活动分片
            $sql = "select db_cluster_id,shard_id from kunlun_metadata_db.shard_nodes where status!='deleted' group by db_cluster_id,shard_id";
            $shards = $this->Cluster_model->getList($sql);
            $error_list = array();

            if (!empty($shards)) {
                foreach ($shards as $shard) {
                    $cluster_id = $shard['db_cluster_id'];
                    $shard_id = $shard['shard_id'];
                    
                    // 获取集群名称
                    $res_cluster = $this->getClusterName($cluster_id);
                    $cluster_name = !empty($res_cluster) ? $res_cluster[0]['nick_name'] : '';
                    $shard_name = $this->getShardName($cluster_id, $shard_id);

                    // 检查计算节点异常
                    $comp_node_errors = $this->compNodeError($cluster_id, $cluster_name);
                    if (!empty($comp_node_errors)) {
                        foreach ($comp_node_errors as $comp_error) {
                            $comp_id = $comp_error['id'];
                            $arr = explode(',', $comp_error);
                            $arr_ip = explode('(', $arr[1]);
                            $ip = $arr_ip[0];
                            $arr_port = explode(')', $arr_ip[1]);
                            $port = $arr_port[0];

                            $alarm_data = [
                                'cluster_id' => $cluster_id,
                                'message' => $comp_error,
                                'ip' => $ip,
                                'port' => $port
                            ];
                            
                            $msg_data = urldecode(json_encode($alarm_data));
                            
                            // 检查是否已存在未处理的告警
                            $select_comp_sql = "select job_info from kunlun_metadata_db.cluster_alarm_info where job_info=? and status='unhandled'";
                            $res_comp_select = $this->Cluster_model->getList($select_comp_sql, [$msg_data]);
                            
                            if (empty($res_comp_select)) {
                               

                                // 记录告警信息
                                $insert_comp_sql = "insert into kunlun_metadata_db.cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,compnodeid) values (?,?,now(),?,?)";
                                $this->Cluster_model->updateList($insert_comp_sql, ['comp_node_exception', $msg_data, $cluster_id, $comp_id]);
                            }
                        }
                    }

                    // 检查主备延迟和存储节点异常
                    if (!empty($cluster_name) && $shard_name) {
                        $errors = $this->shardError($cluster_id, $shard_id, $user_id, $shard_name, $cluster_name);
                        if (!empty($errors)) {
                            foreach ($errors as $error) {
                                $error_list[] = $error;
                                
                                // 处理主备延迟告警
                                if (strpos($error, 'delay') !== false) {
                                    $arr = explode(',', $error);
                                    $arr_ip = explode('(', $arr[1]);
                                    $ip = $arr_ip[0];
                                    $arr_port = explode(')', $arr_ip[1]);
                                    $port = $arr_port[0];
                                    
                                    $msg_data = urldecode(
                                        json_encode(
                                            array(
                                                'cluster_id' => $cluster_id,
                                                'shard_id' => $shard_id,
                                                'message' => urlencode($error),
                                                'ip' => $ip,
                                                'port' => $port
                                            )
                                        )
                                    );
                                    
                                    $select_sql = "select job_info from kunlun_metadata_db.cluster_alarm_info where job_info=? and status='unhandled'";
                                    $res_select = $this->Cluster_model->getList($select_sql, [$msg_data]);
                                    
                                    if (empty($res_select)) {
                                       
                                        
                                        $insert_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,shardid) values (?,?,now(),?,?)";
                                        $this->Cluster_model->updateList($insert_sql, ['standby_delay', $msg_data, $cluster_id, $shard_id]);
                                    }
                                } 
                                // 处理存储节点异常告警
                                else if (strpos($error, 'Storage Node Exception') !== false) {
                                    $arr = explode(',', $error);
                                    $arr_ip = explode('(', $arr[1]);
                                    $ip = $arr_ip[0];
                                    $arr_port = explode(')', $arr_ip[1]);
                                    $port = $arr_port[0];
                                    
                                    $msg_data = urldecode(
                                        json_encode(
                                            array(
                                                'cluster_id' => $cluster_id,
                                                'shard_id' => $shard_id,
                                                'message' => urlencode($error),
                                                'ip' => $ip,
                                                'port' => $port
                                            )
                                        )
                                    );
                                    
                                    $select_storage_sql = "select job_info from cluster_alarm_info where job_info=? and status='unhandled'";
                                    $res_storage_select = $this->Cluster_model->getList($select_storage_sql, [$msg_data]);
                                    
                                    if (empty($res_storage_select)) {
                                        
                                        
                                        $insert_storage_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,shardid) values (?,?,now(),?,?)";
                                        $this->Cluster_model->updateList($insert_storage_sql, ['storage_node_exception', $msg_data, $cluster_id, $shard_id]);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            
            // 检查设备离线
            $machine_errors = $this->machineError();
            if (!empty($machine_errors)) {
                foreach ($machine_errors as $machine_row) {
                    $arr = explode(',', $machine_row);
                    $ip = $arr[0];
                    $msg_data = urldecode(
                        json_encode(
                            array(
                                'message' => urlencode($machine_row),
                                'ip' => $ip
                            )
                        )
                    );
                    
                    $select_machine_sql = "select job_info from cluster_alarm_info where job_info=? and status='unhandled'";
                    $res_machine_select = $this->Cluster_model->getList($select_machine_sql, [$msg_data]);
                    
                    if (empty($res_machine_select)) {
                        
                        $insert_machine_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp) values (?,?,now())";
                        $this->Cluster_model->updateList($insert_machine_sql, ['machine_offline', $msg_data]);
                    }
                }
            }
            
            // 检查主备切换失败
            $manual_switch_errors = $this->manualSwitchError();
            if (!empty($manual_switch_errors)) {
                foreach ($manual_switch_errors as $switch_row) {
                    $switch_cluster_id = $switch_row['cluster_id'];
                    $switch_shard_id = $switch_row['shard_id'];
                    $switch_row_json = json_encode($switch_row);
                    
                    $select_switch_sql = "select job_info from cluster_alarm_info where job_info=?";
                    $res_select_switch = $this->Cluster_model->getList($select_switch_sql, [$switch_row_json]);
                    
                    if (empty($res_select_switch)) {
                       
                        $insert_switch_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,shardid) values (?,?,now(),?,?)";
                        $this->Cluster_model->updateList($insert_switch_sql, ['manual_switch', $switch_row_json, $switch_cluster_id, $switch_shard_id]);
                    }
                }
            }
            
            // 检查搬表失败
            $table_move_errors = $this->tableMoveError();
            if (!empty($table_move_errors)) {
                foreach ($table_move_errors as $move_row) {
                    $move_cluster_id = $move_row['cluster_id'];
                    $move_job_id = $move_row['id'];
                    $move_shard_id = $move_row['src_shard_id'];
                    $move_row_json = json_encode($move_row);
                    
                    $select_move_sql = "select job_id from cluster_alarm_info where job_id=?";
                    $res_move = $this->Cluster_model->getList($select_move_sql, [$move_job_id]);
                    
                    if (empty($res_move)) {
                       
                        $insert_move_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,shardid,job_id) values (?,?,now(),?,?,?)";
                        $this->Cluster_model->updateList($insert_move_sql, ['expand_cluster', $move_row_json, $move_cluster_id, $move_shard_id, $move_job_id]);
                    }
                }
            }
            
            // 检查备份失败
            $backup_errors = $this->backupError();
            if (!empty($backup_errors)) {
                foreach ($backup_errors as $backup_row) {
                    $backup_job_id = $backup_row['id'];
                    if (!empty($backup_row['cluster_id'])) {
                        $backup_cluster_id = $backup_row['cluster_id'];
                        $backup_comp_id = $backup_row['compid'];
                        $backup_shard_id = $backup_row['shardid'];
                        $backup_row_json = json_encode($backup_row);
                        
                        $select_backup_sql = "select job_id from cluster_alarm_info where job_id=?";
                        $res_backup = $this->Cluster_model->getList($select_backup_sql, [$backup_job_id]);
                        
                        if (empty($res_backup)) {
                            
                            $insert_backup_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,job_id,cluster_id,shardid,compnodeid) values (?,?,now(),?,?,?,?)";
                            $this->Cluster_model->updateList($insert_backup_sql, [
                                'shard_coldbackup', 
                                $backup_row_json, 
                                $backup_job_id, 
                                $backup_cluster_id, 
                                $backup_shard_id, 
                                $backup_comp_id
                            ]);
                        }
                    }
                }
            }
            
            // 检查集群操作失败
            $cluster_job_errors = $this->clusterJobError();
            if (!empty($cluster_job_errors)) {
                foreach ($cluster_job_errors as $cluster_row) {
                    $job_id = $cluster_row['id'];
                    $job_type = $cluster_row['job_type'];
                    
                    $cluster_id = null;
                    if (isset($cluster_row['paras']['cluster_id'])) {
                        $cluster_id = $cluster_row['paras']['cluster_id'];
                    }
                    
                    $shard_id = null;
                    if (isset($cluster_row['paras']['shard_id'])) {
                        $shard_id = $cluster_row['paras']['shard_id'];
                    }
                    
                    $cluster_row_json = json_encode($cluster_row);
                    
                    $select_cluster_sql = "select job_id from cluster_alarm_info where job_id=?";
                    $res_cluster = $this->Cluster_model->getList($select_cluster_sql, [$job_id]);
                    
                    if (empty($res_cluster)) {
                        
                        
                        $insert_cluster_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp,cluster_id,shardid,job_id) values (?,?,now(),?,?,?)";
                        $this->Cluster_model->updateList($insert_cluster_sql, [$job_type, $cluster_row_json, $cluster_id, $shard_id, $job_id]);
                    }
                }
            }
            
            // 检查 RCR 错误
            $rcr_errors = $this->rcrError($user_id);
            if (!empty($rcr_errors)) {
                foreach ($rcr_errors as $rcr_row) {
                    // 处理 RCR 同步异常
                    if (strpos($rcr_row, 'synchronous') !== false) {
                        $arr = explode('[', $rcr_row);
                        $db = substr($arr[1], 0, strlen($arr[1]) - 1);
                        $arr_id = explode('RCR', $arr[0]);
                        $rcr_id = explode('of', $arr_id[0]);
                        
                        $msg_data = urldecode(
                            json_encode(
                                array(
                                    'message' => urlencode($rcr_row),
                                    'dump_host' => $db,
                                    'rcr_infos_id' => $rcr_id[1]
                                )
                            )
                        );
                        
                        $select_rcr_sql = "select job_info from cluster_alarm_info where job_info=? and status='unhandled'";
                        $res_rcr_select = $this->Cluster_model->getList($select_rcr_sql, [$msg_data]);
                        
                        if (empty($res_rcr_select)) {
                            
                            $insert_rcr_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp) values (?,?,now())";
                            $this->Cluster_model->updateList($insert_rcr_sql, ['rcr_sync_abnormal', $msg_data]);
                        }
                    }
                    // 处理 RCR 延迟过大
                    else if (strpos($rcr_row, '延迟') !== false) {
                        $arr = explode('[', $rcr_row);
                        $arr_id = explode('的', $arr[0]);
                        $rcr_id = explode('为', $arr_id[0]);
                        
                        $msg_data = urldecode(
                            json_encode(
                                array(
                                    'message' => urlencode($rcr_row),
                                    'rcr_infos_id' => $rcr_id[1]
                                )
                            )
                        );
                        
                        $select_rcr_delay_sql = "select job_info from cluster_alarm_info where job_info=? and status='unhandled'";
                        $res_rcr_delay_select = $this->Cluster_model->getList($select_rcr_delay_sql, [$msg_data]);
                        
                        if (empty($res_rcr_delay_select)) {
                            
                            $insert_rcr_delay_sql = "insert into cluster_alarm_info(alarm_type,job_info,occur_timestamp) values (?,?,now())";
                            $this->Cluster_model->updateList($insert_rcr_delay_sql, ['rcr_delay', $msg_data]);
                        }
                    }
                }
            }

            $data = [
                'code' => 200,
                'message' => '告警处理成功',
                'error_list' => $error_list
            ];
            echo json_encode($data);

        } catch (Exception $e) {
            log_message('error', 'Error in getClusterMonitor: ' . $e->getMessage());
            $data = [
                'code' => 500,
                'message' => $e->getMessage()
            ];
            echo json_encode($data);
        }
    }

    private function validateUser($user_name)
    {
        if (empty($user_name)) {
            return false;
        }

        $sql = "SELECT id FROM kunlun_user WHERE name = ?";
        $result = $this->Login_model->getList($sql, [$user_name]);
        return !empty($result) ? $user_name : false;
    }

    private function sendBatchAlarms($alarms)
    {
        if (empty($alarms)) {
            return;
        }

        // 批量插入告警记录
        $values = [];
        $params = [];
        foreach ($alarms as $alarm) {
            // 获取告警级别
            $level = $this->alarmhandler->getAlarmLevel($alarm['type'], $alarm['content']);
            
            $values[] = "(?, ?, NOW(), ?, ?, ?, ?)";
            $params = array_merge($params, [
                $alarm['type'],
                $alarm['content'],
                $alarm['cluster_id'] ?? null,
                $alarm['shard_id'] ?? null,
                $alarm['job_id'] ?? null,
                $level
            ]);
        }

        $sql = "INSERT INTO cluster_alarm_info 
                (alarm_type, job_info, occur_timestamp, cluster_id, shardid, job_id, alarm_level) 
                VALUES " . implode(',', $values);
        
        if ($this->Cluster_model->updateList($sql, $params)) {
            // 异步发送通知
            foreach ($alarms as $alarm) {
                // 根据配置选择推送方式
                $pushTypes = $this->getPushMessageType($alarm['type']);
                foreach ($pushTypes as $push) {
                    switch ($push['type']) {
                        case 'phone_message':
                            $this->SendSms($push['phone'], $alarm['content']);
                            break;
                        case 'mail':
                            $this->SendMail($push['email'], $alarm['type'], $alarm['content']);
                            break;
                        case 'wechat':
                            $this->notificationhandler->sendWechatMessage($alarm['content']);
                            break;
                    }
                }
            }
        }
    }

    public function getShardName($db_cluster_id, $id)
    {
        $sql = "select name from shards where id='$id' and db_cluster_id='$db_cluster_id' and status!='deleted'";
        $this->load->model('Cluster_model');
        $res = $this->Cluster_model->getList($sql);
        if (!empty($res)) {
            return $res[0]['name'];
        } else {
            return '';
        }
    }

    public function getClusterName($id)
    {
        $sql = "select name,nick_name from db_clusters where id='$id' and status!='deleted'";
        $this->load->model('Cluster_model');
        $res = $this->Cluster_model->getList($sql);
        return $res;
    }

    public function getPushMessageType($alarm_type)
    {
        $sqldalay = "select * from kunlun_metadata_db.cluster_alarm_user where `id`='" . $alarm_type . "'";
        $res = $this->Cluster_model->getList($sqldalay);
        if ($res) {
            $accept_user = $res[0]['accept_user'];
            $resp = [];
            if ($accept_user != "") {
                $accept_user_array = explode(",", $accept_user);
                foreach ($accept_user_array as $pk => $u) {
                    $alarm_type_content = $res[0]['push_type'];
                    $sql_user = "select * from kunlun_dba_tools_db.kunlun_user where id=" . $accept_user_array[$pk];
                    $res_user = $this->Login_model->getList($sql_user);
                    if ($res_user) {
                        $alarm_type_Arr = explode(",", $alarm_type_content);
                        for ($i = 0; $i < count($alarm_type_Arr); $i++) {
                            $tmp = [];
                            $tmp['type'] = $alarm_type_Arr[$i];
                            $tmp['phone'] = $res_user[0]['phone_number'];
                            $tmp['email'] = $res_user[0]['email'];
                            $resp[] = $tmp;
                        }
                    }
                }
            }
            return $resp;
        }
        return [];
    }

    public function SendSms($mobile, $content, $titel = "")
    {
        require_once(APPPATH . "helpers/Sms.php");
        $sqldalay = "select * from kunlun_metadata_db.cluster_alarm_message_config ";
        $push_config = $this->Cluster_model->getList($sqldalay);
        $phone_message_config = [];
        foreach ($push_config as $res) {
            if ($res['type'] == 'phone_message') {
                $phone_message_config = json_decode($res['message'], true);
            }
        }

        if (count($phone_message_config)) {
            $SigId = str_replace('u', '\u', $phone_message_config['SigId']);
            $SigId = $this->decodeUnicode($SigId);
            $cofig = array(
                'accessKeyId' => $phone_message_config['AccessKeyId'],
                'accessKeySecret' => $phone_message_config['SecretKey'],
                'signName' => $SigId,
                'templateCode' => $phone_message_config['TemplateId'],
            );
            $sms = new Sms($cofig);
            if ($sms) {
                if ($cofig['signName'] == "云信发") {
                    $content = "4331";
                }
                //exit($mobile);
                $status = $sms->send_verify($mobile, urlencode($content));

                $log_sql = "INSERT INTO cluster_alarm_push_log (`alarm_type`, `push_type`, `content`, `content_res`, `create_at`) VALUES ( ?,?,?,?,?)";
                $this->Cluster_model->updateList($log_sql, [$titel, 'phone_message', $content, $status, time()]);

                return $status;
            } else {
                return true;
            }
        }
    }

    public function SendMail($mailAddress, $titel, $content)
    {
        log_message('debug', '=== SendMail() Start ===');
        log_message('debug', 'Mail address: ' . $mailAddress);
        log_message('debug', 'Title: ' . $titel);
        log_message('debug', 'Content: ' . $content);

        require_once(APPPATH . "helpers/SendMail.php");
        $sqldalay = "select * from cluster_alarm_message_config ";
        $push_config = $this->Cluster_model->getList($sqldalay);
        $email_config = [];
        foreach ($push_config as $res) {
            if ($res['type'] == 'email') {
                $email_config = json_decode($res['message'], true);
            }
        }
        log_message('debug', 'Email config: ' . json_encode($email_config));

        $SendMail = new SendMail($email_config['AccessKeyId'], $email_config['SecretKey'], $email_config['AccountName']);
        if ($SendMail) {
            $SendMail->ToAddress = $mailAddress;
            $SendMail->FromAlias = "kunlunbase";
            $SendMail->TagName = "kunlunbase";
            $SendMail->Subject = $titel;
            $SendMail->HtmlBody = $content;
            $EmailRes = $SendMail->send();
            log_message('debug', 'Email send result: ' . json_encode($EmailRes));

            $log_sql = "INSERT INTO cluster_alarm_push_log (`alarm_type`, `push_type`, `content`, `content_res`, `create_at`) VALUES ( ?,?,?,?,?)";
            $this->Cluster_model->updateList($log_sql, [$titel, 'email', $content, $EmailRes, time()]);

            log_message('debug', '=== SendMail() End ===');
            return $EmailRes;
        } else {
            log_message('error', 'Failed to initialize SendMail');
            return false;
        }
    }

    public function decodeUnicode($str)
    {
        return preg_replace_callback(
            "#\\\u([0-9a-f]{4})#i",
            function ($r) {
                return iconv('UCS-2BE', 'UTF-8', pack('H4', $r[1]));
            },
            $str
        );
    }

    public function sendTestEmail()
    {
        log_message('debug', '=== sendTestEmail() Start ===');
        
        // 获取POST数据
        $post_data = json_decode(file_get_contents('php://input'), true);
        log_message('debug', 'Post data: ' . json_encode($post_data));
        
        try {
            // 验证必要参数
            if (empty($post_data['email']) || empty($post_data['content']) || empty($post_data['mail_config'])) {
                throw new Exception('Missing required parameters');
            }

            $mail_config = $post_data['mail_config'];
            $to_email = $post_data['email'];
            $content = $post_data['content'];

            $mail = new PHPMailer(true);

            try {
                // 服务器设置
                $mail->SMTPDebug = 0; // 启用详细的调试输出
                $mail->isSMTP();
                $mail->Host = $mail_config['stmp']; 
                $mail->SMTPAuth = true;
                $mail->Username = $mail_config['user_name'];
                $mail->Password = $mail_config['password'];
                
                // 根据不同的邮箱服务商设置不同的加密方式和端口
                if (strpos($mail_config['stmp'], 'qq.com') !== false) {
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS; // SSL加密
                    $mail->Port = 465;
                } else if (strpos($mail_config['stmp'], '163.com') !== false) {
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS; // TLS加密
                    $mail->Port = 587;
                } else if (strpos($mail_config['stmp'], 'gmail.com') !== false) {
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
                    $mail->Port = 587;
                } else {
                    // 默认设置
                    $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
                    $mail->Port = 465;
                }

                // 设置超时时间
                $mail->Timeout = 10;
                $mail->SMTPOptions = array(
                    'ssl' => array(
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true
                    )
                );

                // 发件人和收件人
                $mail->setFrom($mail_config['user_name']);
                $mail->addAddress($to_email);

                // 邮件内容
                $mail->isHTML(true);
                $mail->Subject = 'Test Email';
                $mail->Body = $content;
                $mail->CharSet = 'UTF-8'; // 设置编码

                // 发送邮件
                $result = $mail->send();
                
                // 记录推送日志
                $log_sql = "INSERT INTO cluster_alarm_push_log (`alarm_type`, `push_type`, `content`, `content_res`, `create_at`) VALUES (?, ?, ?, ?, ?)";
                $this->Cluster_model->updateList($log_sql, [
                    'test_email',
                    'email',
                    $content,
                    $result ? 'success' : 'failed',
                    time()
                ]);

                if ($result) {
                    $data = [
                        'code' => 200,
                        'message' => 'Email sent successfully'
                    ];
                } else {
                    $data = [
                        'code' => 500,
                        'message' => 'Failed to send email'
                    ];
                }

            } catch (Exception $e) {
                log_message('error', 'PHPMailer Error: ' . $e->getMessage());
                // 记录详细的错误信息
                log_message('error', 'SMTP Debug: ' . $mail->ErrorInfo);
                throw new Exception('Failed to send email: ' . $e->getMessage() . '. SMTP Error: ' . $mail->ErrorInfo);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Send test email failed: ' . $e->getMessage());
            $data = [
                'code' => 500,
                'message' => $e->getMessage()
            ];
        }
        
        log_message('debug', 'Response: ' . json_encode($data));
        log_message('debug', '=== sendTestEmail() End ===');
        
        echo json_encode($data);
    }

    private function processShardAlarms($shard, $user_name, $shard_name, $cluster_name, &$alarms)
    {
        // $errors = $this->shardError(
        //     $shard['db_cluster_id'],
        //     $shard['shard_id'],
        //     $user_name
        // );
        // if (!empty($errors)) {
        //     foreach ($errors as $error) {
        //         if ($this->isNewAlarm('shard_exception', $error)) {
        //             $alarms[] = [
        //                 'type' => 'shard_exception',
        //                 'content' => $error,
        //             ];
        //         }
        //     }
        // }
    }

    private function recordMetrics($type, $data) {
        $sql = "INSERT INTO cluster_performance_metrics (metric_type, metric_data, create_at) VALUES (?, ?, ?)";
        $this->Cluster_model->updateList($sql, [
            $type,
            json_encode($data),
            time()
        ]);
    }

    public function monitorPerformance($alarm_id, $push_type) {
        // 监控告警处理性能
        // $this->monitorAlarmProcessing($alarm_id);
        // 监控推送性能
        // $this->monitorPushPerformance($push_type);
    }

    private function monitorAlarmProcessing($alarm_id) {
        // try {
        //     $metrics = $this->getAlarmMetrics($alarm_id);
        //     if ($metrics) {
        //         $this->recordMetrics('alarm_processing', $metrics);
        //         $this->checkPerformanceThreshold('alarm_processing', $metrics);
        //     }
        // } catch (Exception $e) {
        //     log_message('error', 'Failed to monitor alarm processing: ' . $e->getMessage());
        // }
    }

    private function getAlarmMetrics($alarm_id) {
        $sql = "SELECT id, occur_timestamp, COALESCE(handle_time, NOW()) as end_time,
                       TIMESTAMPDIFF(SECOND, occur_timestamp, COALESCE(handle_time, NOW())) as duration
                FROM cluster_alarm_info WHERE id = ?";
        $result = $this->Cluster_model->getList($sql, [$alarm_id]);
        return !empty($result) ? [
            'alarm_id' => $alarm_id,
            'duration' => $result[0]['duration'],
            'timestamp' => time()
        ] : null;
    }

    private function monitorPushPerformance($push_type) {
        try {
            $metrics = $this->getPushMetrics($push_type);
            if ($metrics) {
                $this->recordMetrics('push_performance', $metrics);
                $this->checkPerformanceThreshold('push_performance', $metrics);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to monitor push performance: ' . $e->getMessage());
        }
    }

    private function getPushMetrics($push_type) {
        $sql = "SELECT COUNT(*) as total,
                       SUM(CASE WHEN content_res = 'success' THEN 1 ELSE 0 END) as success
                FROM cluster_alarm_push_log WHERE push_type = ?
                AND create_at > UNIX_TIMESTAMP(DATE_SUB(NOW(), INTERVAL 5 MINUTE))";
        $result = $this->Cluster_model->getList($sql, [$push_type]);
        if (!empty($result)) {
            return [
                'push_type' => $push_type,
                'total' => $result[0]['total'],
                'success' => $result[0]['success'],
                'success_rate' => $result[0]['total'] > 0 ? ($result[0]['success'] / $result[0]['total']) * 100 : 100,
                'timestamp' => time()
            ];
        }
        return null;
    }

    private function checkPerformanceThreshold($type, $metrics) {
        // 这里添加性能阈值检查的逻辑
    }

    public function getLoginModel() {
        return $this->Login_model;
    }

    private function shardError($cluster_id, $shard_id, $user_id, $shard_name, $cluster_name)
    {
        log_message('debug', "检查集群 {$cluster_name} 分片 {$shard_name} 的错误");
        
        $errors = [];
        
        // 检查主备延迟
        $delay_errors = $this->checkReplicationDelay($cluster_id, $shard_id, $cluster_name, $shard_name, $user_id);
        if (!empty($delay_errors)) {
            $errors = array_merge($errors, $delay_errors);
        }
        
        // 检查存储节点异常
        $node_errors = $this->checkStorageNodeErrors($cluster_id, $shard_id, $cluster_name, $shard_name);
        if (!empty($node_errors)) {
            $errors = array_merge($errors, $node_errors);
        }
        
        return $errors;
    }

    /**
     * 检查主备复制延迟
     * 
     * @param int $cluster_id 集群ID
     * @param int $shard_id 分片ID
     * @param string $cluster_name 集群名称
     * @param string $shard_name 分片名称
     * @param int $user_id 用户ID
     * @return array 延迟错误列表
     */
    private function checkReplicationDelay($cluster_id, $shard_id, $cluster_name, $shard_name, $user_id)
    {
        log_message('debug', "检查集群 {$cluster_name} 分片 {$shard_name} 的复制延迟");
        
        // 查询最大延迟值
        $select_sql = "select max_delay_time from shard_max_dalay where shard_id=? and db_cluster_id=? and user_id=?";
        $res_select = $this->Cluster_model->getList($select_sql, [$shard_id, $cluster_id, $user_id]);
        $maxtime = !empty($res_select) ? $res_select[0]['max_delay_time'] : '100';
        
        // 查询延迟超过最大值的节点
        $sql = "select hostaddr, port, replica_delay from shard_nodes where shard_id=? and db_cluster_id=? and replica_delay>? and status!='deleted'";
        $res = $this->Cluster_model->getList($sql, [$shard_id, $cluster_id, $maxtime]);
        
        $errors = [];
        if (!empty($res)) {
            foreach ($res as $row) {
                $errors[] = $cluster_name . '(' . $cluster_id . '),' . $shard_name . ',' . $row['hostaddr'] . '(' . $row['port'] . ')' . 'The current delay is' . $row['replica_delay'] . 's,Exceed delay maximum' . $maxtime . 's';
            }
        }
        
        return $errors;
    }

    /**
     * 检查存储节点异常
     * 
     * @param int $cluster_id 集群ID
     * @param int $shard_id 分片ID
     * @param string $cluster_name 集群名称
     * @param string $shard_name 分片名称
     * @return array 存储节点错误列表
     */
    private function checkStorageNodeErrors($cluster_id, $shard_id, $cluster_name, $shard_name)
    {
        log_message('debug', "检查集群 {$cluster_name} 分片 {$shard_name} 的存储节点状态");
        
        // 查询状态为 inactive 的存储节点
        $sql_storage = "select hostaddr, port from shard_nodes where shard_id=? and db_cluster_id=? and status='inactive'";
        $res_storage = $this->Cluster_model->getList($sql_storage, [$shard_id, $cluster_id]);
        
        $errors = [];
        if (!empty($res_storage)) {
            foreach ($res_storage as $row_stor) {
                $errors[] = $cluster_name . '(' . $cluster_id . '),' . $shard_name . ',' . $row_stor['hostaddr'] . '(' . $row_stor['port'] . ')' . 'Storage Node Exception';
            }
        }
        
        return $errors;
    }

   
    /**
     * 检查计算节点异常
     * 
     * @param int $cluster_id 集群ID
     * @param string $cluster_name 集群名称
     * @return array 异常计算节点列表
     */
    private function compNodeError($cluster_id, $cluster_name)
    {
        log_message('debug', "检查集群 {$cluster_name}(ID: {$cluster_id}) 的计算节点异常");
        
        // 查询计算节点状态
        $sql = "SELECT id, hostaddr, port, status FROM comp_nodes 
                WHERE db_cluster_id = ? AND status != 'deleted'";
        $comp_nodes = $this->Cluster_model->getList($sql, [$cluster_id]);
        
        $error_nodes = [];
        
        if (!empty($comp_nodes)) {
            foreach ($comp_nodes as $node) {
                // 检查节点是否在线
                $is_online = $this->checkNodeOnline($node['hostaddr'], $node['port']);
                
                if (!$is_online) {
                    $error_message = "计算节点异常: 集群 {$cluster_name}, 节点 {$node['hostaddr']}({$node['port']}) 无法连接";
                    log_message('error', $error_message);
                    
                    $error_nodes[] = [
                        'id' => $node['id'],
                        'message' => $error_message,
                        'hostaddr' => $node['hostaddr'],
                        'port' => $node['port']
                    ];
                }
            }
        }
        
        return $error_nodes;
    }
    
    /**
     * 检查节点是否在线
     * 
     * @param string $host 主机地址
     * @param int $port 端口
     * @return bool 节点是否在线
     */
    private function checkNodeOnline($host, $port)
    {
        // 使用 socket 连接检查节点是否在线
        $socket = @fsockopen($host, $port, $errno, $errstr, 5);
        if ($socket) {
            fclose($socket);
            return true;
        }
        return false;
    }

    /**
     * 检查设备离线
     * 
     * @return array 离线设备列表
     */
    public function machineError()
    {
        log_message('debug', "检查设备离线状态");
        
        $sql = "SELECT hostaddr FROM server_nodes WHERE node_stats='dead' GROUP BY hostaddr";
        $res = $this->Cluster_model->getList($sql);
        
        $error = array();
        if (!empty($res)) {
            foreach ($res as $row) {
                $error[] = $row['hostaddr'] . ',The device is offline';
            }
        }
        
        return $error;
    }

    /**
     * 检查主备切换失败
     * 
     * @return array 主备切换失败列表
     */
    public function manualSwitchError()
    {
        log_message('debug', "检查主备切换失败");
        
        $sql = "SELECT host, shard_id, cluster_id, taskid, err_msg as message, consfailover_msg 
                FROM rbr_consfailover WHERE err_code != 0";
        $res = $this->Cluster_model->getList($sql);
        
        foreach ($res as $row => $value) {
            $string = preg_replace('/\r|\n/', '\n', trim($res[$row]['consfailover_msg']));
            $string = json_decode($string, true);
            
            if (!empty($string['sw_type'])) {
                $res[$row]['sw_type'] = $string['sw_type'];
            } else {
                $res[$row]['sw_type'] = '';
            }
            
            unset($res[$row]['consfailover_msg']);
        }
        
        return $res;
    }

    /**
     * 检查搬表失败
     * 
     * @return array 搬表失败列表
     */
    public function tableMoveError()
    {
        log_message('debug', "检查搬表失败");
        
        $sql = "SELECT id, when_started, when_ended, memo, job_info, user_name 
                FROM cluster_general_job_log 
                WHERE status='failed' AND job_type='expand_cluster'";
        $res = $this->Cluster_model->getList($sql);
        
        foreach ($res as $row => $value) {
            // 处理 memo 字段
            $string = preg_replace('/\r|\n/', '\n', trim($res[$row]['memo']));
            $string = json_decode($string, true);
            if (!empty($string['error_info'])) {
                $res[$row]['message'] = $string['error_info'];
            } else {
                $res[$row]['message'] = '';
            }
            
            // 处理 job_info 字段
            $job_info = preg_replace('/\r|\n/', '\n', trim($res[$row]['job_info']));
            $job_info = json_decode($job_info, true);
            if (!empty($job_info)) {
                $res[$row]['cluster_id'] = $job_info['paras']['cluster_id'];
                $res[$row]['drop_old_table'] = $job_info['paras']['drop_old_table'];
                $res[$row]['dst_shard_id'] = $job_info['paras']['dst_shard_id'];
                $res[$row]['src_shard_id'] = $job_info['paras']['src_shard_id'];
                $res[$row]['table_list'] = $job_info['paras']['table_list'];
            }
            
            // 移除原始字段
            unset($res[$row]['memo']);
            unset($res[$row]['job_info']);
        }
        
        return $res;
    }

    /**
     * 检查备份失败
     * 
     * @return array 备份失败列表
     */
    public function backupError()
    {
        log_message('debug', "检查备份失败");
        
        $sql = "SELECT id, job_type, memo, when_started, when_ended, job_info, user_name 
                FROM cluster_general_job_log 
                WHERE status='failed' AND job_type='shard_coldbackup'";
        $res = $this->Cluster_model->getList($sql);
        
        foreach ($res as $row => $value) {
            // 处理 memo 字段
            $string = preg_replace('/\r|\n/', '\n', trim($res[$row]['memo']));
            $string = json_decode($string, true);
            if (!empty($string['error_info'])) {
                $res[$row]['message'] = $string['error_info'];
            } else {
                $res[$row]['message'] = '';
            }
            
            // 处理 job_info 字段
            $job_info = preg_replace('/\r|\n/', '\n', trim($res[$row]['job_info']));
            $job_info = json_decode($job_info, true);
            if (!empty($job_info)) {
                // 获取节点信息
                $if_exist = $this->getIpPort($job_info['paras']['ip'], $job_info['paras']['port']);
                if ($if_exist) {
                    if ($if_exist[0]['type'] == 'compute') {
                        $res[$row]['ip'] = $job_info['paras']['ip'];
                        $res[$row]['port'] = $job_info['paras']['port'];
                        $res[$row]['compid'] = $if_exist[0]['id'];
                        $res[$row]['shardid'] = null;
                        $res[$row]['cluster_id'] = $if_exist[0]['db_cluster_id'];
                    } else {
                        $res[$row]['ip'] = $job_info['paras']['ip'];
                        $res[$row]['port'] = $job_info['paras']['port'];
                        $res[$row]['compid'] = null;
                        $res[$row]['shardid'] = $if_exist[0]['id'];
                        $res[$row]['cluster_id'] = $if_exist[0]['db_cluster_id'];
                    }
                } else {
                    $res[$row]['ip'] = '';
                    $res[$row]['port'] = '';
                    $res[$row]['compid'] = '';
                    $res[$row]['shardid'] = '';
                    $res[$row]['cluster_id'] = '';
                }
            }
            
            // 移除原始字段
            unset($res[$row]['memo']);
            unset($res[$row]['job_info']);
        }
        
        return $res;
    }

    /**
     * 检查集群操作失败
     * 
     * @return array 集群操作失败列表
     */
    public function clusterJobError()
    {
        log_message('debug', "检查集群操作失败");
        
        $sql = "SELECT id, job_type, memo, when_started, when_ended, job_info, user_name 
                FROM cluster_general_job_log 
                WHERE status='failed' AND job_type IN (
                    'create_cluster', 'delete_cluster', 'add_shards', 'delete_shard', 
                    'add_comps', 'delete_comp', 'add_nodes', 'delete_node', 'manual_backup_cluster'
                )";
        $res = $this->Cluster_model->getList($sql);
        
        foreach ($res as $row => $value) {
            // 处理 memo 字段
            $string = preg_replace('/\r|\n/', '\n', trim($res[$row]['memo']));
            $string = json_decode($string, true);
            if (!empty($string['error_info'])) {
                $res[$row]['message'] = $string['error_info'];
            } else {
                $res[$row]['message'] = '';
            }
            
            // 处理 job_info 字段
            $job_info = preg_replace('/\r|\n/', '\n', trim($res[$row]['job_info']));
            $job_info = json_decode($job_info, true);
            if (!empty($job_info['paras'])) {
                $res[$row]['paras'] = $job_info['paras'];
            } else {
                $res[$row]['paras'] = '';
            }
            
            // 移除原始字段
            unset($res[$row]['memo']);
            unset($res[$row]['job_info']);
        }
        
        return $res;
    }

    /**
     * 检查 RCR 错误
     * 
     * @param int $user_id 用户ID
     * @return array RCR 错误列表
     */
    public function rcrError($user_id): array
    {
        log_message('debug', "检查 RCR 错误，用户ID: {$user_id}");
        
        $error = array();
        
        // 检查 RCR 同步异常
        $sql = "SELECT rcr_infos_id, dump_host 
                FROM cluster_rcr_meta_sync 
                WHERE meta_sync_state = 'disconnect' 
                GROUP BY id";
        $res = $this->Cluster_model->getList($sql);
        
        if (!empty($res)) {
            foreach ($res as $row) {
                $error[] = 'Id of ' . $row['rcr_infos_id'] . 'RCR synchronous abnormal relationships [' . $row['dump_host'] . ']';
            }
        }
        
        // 检查 RCR 延迟过大
        $sqldalay = "SELECT TABLE_NAME 
                     FROM information_schema.TABLES 
                     WHERE TABLE_NAME = 'rcr_max_dalay'";
        $resdalay = $this->Login_model->getList($sqldalay);
        
        if ($resdalay !== false) {
            $select_sql = "SELECT id, rcr_id, max_delay_time 
                           FROM rcr_max_dalay 
                           WHERE user_id = ?";
            $res_select = $this->Login_model->getList($select_sql, [$user_id]);
            
            if (!empty($res_select)) {
                foreach ($res_select as $row_select) {
                    $maxtime = $row_select['max_delay_time'];
                    $rcr_id = $row_select['rcr_id'];
                    
                    $sql_rcr_delay = "SELECT id, replica_delay 
                                      FROM cluster_rcr_infos 
                                      WHERE status != 'deleted' AND id = ?";
                    $res_rcr_delay = $this->Cluster_model->getList($sql_rcr_delay, [$rcr_id]);
                    
                    if (!empty($res_rcr_delay)) {
                        if ($maxtime < $res_rcr_delay[0]['replica_delay']) {
                            $error[] = 'Witch id ' . $rcr_id . 'RCR relationship too much delay, the current delay time for ' . $res_rcr_delay[0]['replica_delay'] . '，最大延迟时间为' . $maxtime;
                        }
                    }
                }
            }
        }
        
        return $error;
    }

    /**
     * 根据 IP 和端口获取节点信息
     * 
     * @param string $ip IP地址
     * @param int $port 端口
     * @return array|string 节点信息或空字符串
     */
    public function getIpPort($ip, $port)
    {
        log_message('debug', "根据 IP: {$ip} 和端口: {$port} 获取节点信息");
        
        // 查询计算节点
        $sql = "SELECT db_cluster_id, id 
                FROM comp_nodes 
                WHERE hostaddr = ? AND port = ? AND status != 'deleted'";
        $res = $this->Cluster_model->getList($sql, [$ip, $port]);
        
        if (!empty($res)) {
            foreach ($res as $row => $value) {
                $res[$row]['type'] = 'compute';
            }
            return $res;
        } else {
            // 查询存储节点
            $sql_shard = "SELECT db_cluster_id, shard_id as id 
                          FROM shard_nodes 
                          WHERE hostaddr = ? AND port = ? AND status != 'deleted'";
            $res_shard = $this->Cluster_model->getList($sql_shard, [$ip, $port]);
            
            if (!empty($res_shard)) {
                foreach ($res_shard as $row1 => $value1) {
                    $res_shard[$row1]['type'] = 'storage';
                }
                return $res_shard;
            } else {
                return '';
            }
        }
    }

    /**
     * 测试生成告警信息
     * 该函数从 AlarmCategory_model 获取所有告警类型，并生成模拟告警信息写入数据库
     * 
     * 注意：此函数仅用于测试目的，模拟生成各种类型的告警数据
     */
    public function testGenerateAlarms()
    {
        header('Content-Type: application/json;charset=utf-8');
        
        try {
            // 加载 AlarmCategory_model
            $this->load->model('AlarmCategory_model');
            
            // 获取所有告警类型详情
            $alarmTypes = $this->AlarmCategory_model->getAlarmTypeDetails();
            
            if (empty($alarmTypes)) {
                throw new Exception('无法获取告警类型详情');
            }
            
            // 模拟集群和分片 ID
            $mockClusterId = 1;
            $mockShardId = 1;
            $mockCompNodeId = 1;
            $mockSvrid = 1;
            
            // 模拟生成的告警数量计数
            $generatedCount = 0;
            $errors = [];
            
            // 为每种告警类型生成一条模拟数据
            foreach ($alarmTypes as $type => $details) {
                // 构建告警信息
                $alarmLevel = $details['level'] ?? 'WARNING';
                
                // 构建模拟的告警详情
                $jobInfo = json_encode([
                    'message' => '这是' . $details['label'] . '的模拟告警信息',
                    'trigger_condition' => $details['trigger_condition'] ?? '',
                    'monitor_metric' => $details['monitor_metric'] ?? '',
                    'detail' => '模拟告警详情 - ' . date('Y-m-d H:i:s'),
                    'mock_data' => true
                ]);
                
                // 准备插入语句
                $insertSql = "INSERT INTO cluster_alarm_info (
                    alarm_level, 
                    alarm_type, 
                    status, 
                    occur_timestamp, 
                    job_info, 
                    cluster_id, 
                    shardid, 
                    compnodeid, 
                    svrid
                ) VALUES (?, ?, 'unhandled', NOW(), ?, ?, ?, ?, ?)";
                
                // 设置参数
                $params = [
                    $alarmLevel,
                    $type,
                    $jobInfo,
                    // 模拟不同告警类型可能关联不同资源
                    in_array($type, ['standby_delay', 'storage_node_exception']) ? $mockClusterId : null,
                    in_array($type, ['standby_delay', 'storage_node_exception']) ? $mockShardId : null,
                    in_array($type, ['comp_node_exception']) ? $mockCompNodeId : null,
                    in_array($type, ['machine_offline']) ? $mockSvrid : null
                ];
                
                // 执行插入
                $result = $this->Cluster_model->updateList($insertSql, $params);
                
                // 检查插入结果
                if (is_array($result) && isset($result['code'])) {
                    $errors[] = "插入告警类型 {$type} 失败: " . ($result['error'] ?? '未知错误');
                } else {
                    $generatedCount++;
                }
            }
            
            // 返回结果
            $response = [
                'success' => true,
                'message' => "成功生成 {$generatedCount} 条模拟告警数据",
                'total_types' => count($alarmTypes),
                'errors' => $errors
            ];
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            $response = [
                'success' => false,
                'message' => '生成告警数据时发生错误: ' . $e->getMessage()
            ];
            
            echo json_encode($response);
        }
    }
}
