<?php

class Cluster_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->db = $this->load->database('default', true);
        $this->config->load('myconfig');
        $this->pg_database = $this->config->item('pg_database');
        $this->pg_username = $this->config->item('pg_username');
        $this->default_username = $this->config->item('default_username');
        $this->post_url = $this->config->item('post_url');
        $this->grafana_key = $this->config->item('grafana_key');
    }

    // 获取 cluster_steam_task 表中的记录
    public function getClusterSteamTask($conditions = [], $limit = 10, $offset = 0)
    {

        // 构建 SQL 查询
        $this->db->select('*');
        $this->db->from('cluster_steams');

        // 添加条件
        if (!empty($conditions)) {
            foreach ($conditions as $key => $value) {
                $this->db->where($key, $value);
            }
        }
        $this->db->order_by('id desc');
        $this->db->limit($limit, $offset);

        $query = $this->db->get();

        return $query->result_array();
    }

    //查询数据
    public function getList($sql, $params = [])
    {
        try {
            // 执行带参数的 SQL 语句
            $q = $this->db->query($sql, $params);
            if ($q && $q->num_rows() > 0) {
                return $q->result_array();
            } else {
                return false;
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }


    public function updateList($sql, $params = [])
    {
        try {
            // 执行带参数的 SQL 语句
            $query = $this->db->query($sql, $params);
            if ($query === false) {
                // 记录错误并返回错误信息
                return $this->db->error();
            }
            return $this->db->affected_rows();
        } catch (Exception $e) {
            // 处理异常并返回错误代码
            return [
                'code' => 500,
                'error' => $e->getMessage()
            ];
        }
    }


    public function create_uuid($prefix = ""): string
    {
        if (PATH_SEPARATOR == ':') {
            return $this->create_uuid(1);
        } else {
            $chars = md5(uniqid(mt_rand(), true));
            $uuid = substr($chars, 0, 8) . '-'
                . substr($chars, 8, 4) . '-'
                . substr($chars, 12, 4) . '-'
                . substr($chars, 16, 4) . '-'
                . substr($chars, 20, 12);
            return $prefix . $uuid;
        }
    }


    public function postData($post_data, $url, $max_recursion = 3)
    {
        // 开始记录请求日志
        log_message('debug', '====== POST REQUEST STARTED ======');
        log_message('debug', 'Function postData called with URL: ' . $url);
        
       
        
        $start_time = microtime(true);
        
        $ch = curl_init($url);
        log_message('debug', 'cURL initialized');
        
        // 设置cURL选项
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);   // 设置超时时间为60秒
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0); // 强制协议为1.0
        curl_setopt($ch, CURLOPT_IPRESOLVE, CURL_IPRESOLVE_V4); // 强制使用IPV4
        
        // 合并所有HTTP头设置，避免覆盖
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Expect: ',
            'Content-Type: application/json',
            'Content-Length: ' . strlen($post_data),
        ]);
        
        // 记录所有cURL选项
        log_message('debug', 'cURL options set: TIMEOUT=60, HTTP_VERSION=1.0, IPRESOLVE=IPV4');
        log_message('debug', 'Headers: Content-Type=application/json, Content-Length=' . strlen($post_data) . ', Expect: ');

        // 日志记录请求的URL
        log_message('info', 'Request URL: ' . $url);
        
        log_message('debug', 'Executing cURL request...');
        $result = curl_exec($ch);
        $end_time = microtime(true);
        $execution_time = number_format($end_time - $start_time, 4);
        
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_errno = curl_errno($ch);
        $curl_error = curl_error($ch);
        
        // 记录cURL请求信息
        log_message('debug', 'cURL request completed in ' . $execution_time . ' seconds');
        log_message('debug', 'HTTP status code: ' . $http_code);

        // 处理错误情况
        if ($curl_errno) {
            // 记录详细的CURL错误信息
            log_message('error', 'CURL Error #' . $curl_errno . ': ' . $curl_error);
            log_message('debug', 'Connection details - connect_time: ' . curl_getinfo($ch, CURLINFO_CONNECT_TIME) . 's, total_time: ' . curl_getinfo($ch, CURLINFO_TOTAL_TIME) . 's');
            
            // 在关闭连接前获取更多信息
            $info = curl_getinfo($ch);
            curl_close($ch);
            
            // 如果已达到最大递归深度，直接返回错误
            if ($max_recursion <= 0) {
                log_message('error', 'Maximum recursion depth reached, giving up');
                log_message('debug', '====== POST REQUEST ENDED WITH ERROR ======');
                return json_encode([
                    'status' => 'connection failed',
                    'error_info' => 'Maximum retries exceeded: ' . $url
                ]);
            }
            
            return $this->handleConnectionFailure($post_data, $url, $max_recursion - 1);
        }

        // 获取更多的CURL信息用于调试
        $info = curl_getinfo($ch);
        log_message('debug', 'cURL info: ' . json_encode($info));
        log_message('debug', 'Total time: ' . $info['total_time'] . 's, Connect time: ' . $info['connect_time'] . 's');
        
        curl_close($ch);
        log_message('debug', 'cURL connection closed');

        // 日志记录返回的结果（截断过长的结果）
        $truncated_result = (strlen($result) > 1000) ? substr($result, 0, 1000) . '... [truncated]' : $result;
        log_message('info', 'Response received');
        log_message('debug', 'Truncated response for logging: ' . $truncated_result);
        log_message('debug', '====== POST REQUEST COMPLETED SUCCESSFULLY ======');

        return $result;
    }
    
    /**
     * 处理连接失败的情况，尝试找到备用节点
     * 
     * @param string $post_data POST数据
     * @param string $url 原始URL
     * @param int $max_recursion 最大递归次数
     * @return string JSON格式的响应
     */
    private function handleConnectionFailure($post_data, $url, $max_recursion)
    {
        log_message('debug', 'Attempting fallback connection strategy...');

        // 解析原始URL获取主机和端口
        $url_parts = parse_url($url);
        if (!isset($url_parts['host']) || !isset($url_parts['port'])) {
            $old_url = explode('/', $url);
            $host_port = isset($old_url[2]) ? $old_url[2] : 'unknown';
        } else {
            $host_port = $url_parts['host'] . ':' . $url_parts['port'];
        }
        
        log_message('debug', 'Original URL host:port: ' . $host_port);
        
        // 查询可用的cluster_mgr节点
        $sql = "select hostaddr,port from kunlun_metadata_db.cluster_mgr_nodes";
        log_message('debug', 'Executing SQL: ' . $sql);
        
        $res = $this->getList($sql);
        $total = count($res);
        log_message('debug', 'Found ' . $total . ' cluster_mgr nodes');
        
        if ($total >= 1) {
            log_message('debug', 'Available cluster_mgr nodes: ' . json_encode($res));
        }

        // 单节点情况处理
        if ($total == 1) {
            $node_host_port = $res[0]['hostaddr'] . ':' . $res[0]['port'];
            log_message('debug', 'Single node scenario: ' . ($host_port == $node_host_port ? 'URL matches' : 'URL does not match') . ' the only available node');
            
            $error_response = [
                'status' => 'connection failed',
                'error_info' => 'Interface request timeout: ' . $host_port
            ];
            log_message('error', 'Interface request timeout for URL: ' . $host_port);
            log_message('debug', '====== POST REQUEST ENDED WITH ERROR ======');
            return json_encode($error_response);
        } 
        // 多节点情况处理
        else if ($total > 1) {
            log_message('debug', 'Multiple nodes scenario: trying each available node');
            
            // 尝试连接每个节点
            foreach ($res as $k => $v) {
                $current_node = $v['hostaddr'] . ':' . $v['port'];
                log_message('debug', 'Trying node #' . ($k + 1) . ': ' . $current_node);
                
                $cluster_mgr_result = $this->getClusterMgrMaster($v['hostaddr'], $v['port']);
                
                if ($cluster_mgr_result === true) {
                    // 成功连接到节点
                    log_message('debug', 'Successfully connected to ClusterMgr: ' . $current_node);
                    $post_url = 'http://' . $current_node . '/HttpService/Emit';
                    log_message('debug', 'Rewriting config with new URL: ' . $post_url);
                    
                    rewrite_config([
                        'post_url' => $post_url,
                    ]);
                    log_message('info', 'Switching to new ClusterMgr URL: ' . $post_url);
                    
                    // 使用新URL重试请求
                    return $this->postData($post_data, $post_url, $max_recursion);
                } 
                // 最后一个节点也失败
                else if ($k + 1 == $total) {
                    log_message('debug', 'Final node check failed. All nodes unreachable.');
                    $error_response = [
                        'status' => 'connection failed',
                        'error_info' => 'Check whether cluster_mgr_nodes table ports in the metadata can send HTTP requests: ' . $current_node
                    ];
                    log_message('error', 'All ClusterMgr nodes failed, last checked: ' . $current_node);
                    log_message('debug', '====== POST REQUEST ENDED WITH ERROR ======');
                    return json_encode($error_response);
                } 
                // 尝试下一个节点
                else {
                    log_message('debug', 'Node check failed, continuing to next node');
                    continue;
                }
            }
        } 
        // 无节点情况处理
        else {
            log_message('debug', 'No cluster_mgr nodes found in database');
            $error_response = [
                'status' => 'connection failed',
                'error_info' => 'Interface request timeout: ' . $host_port
            ];
            log_message('error', 'Interface request timeout for URL: ' . $host_port);
            log_message('debug', '====== POST REQUEST ENDED WITH ERROR ======');
            return json_encode($error_response);
        }
        
        // 如果执行到这里，说明没有找到可用节点
        return json_encode([
            'status' => 'connection failed',
            'error_info' => 'No available cluster_mgr nodes'
        ]);
    }
    
    /**
     * 对敏感数据进行脱敏处理以便安全记录日志
     * 
     * @param string $data 原始数据
     * @return string 脱敏后的数据
     */
    private function sanitizeDataForLogging($data)
    {
        // 检查是否是JSON字符串
        $json_data = json_decode($data, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($json_data)) {
            // 敏感字段列表，根据实际情况可以扩展
            $sensitive_fields = ['password', 'pwd', 'passwd', 'token', 'secret', 'key'];
            
            foreach ($sensitive_fields as $field) {
                if (isset($json_data[$field])) {
                    $json_data[$field] = '***REDACTED***';
                }
            }
            
            return json_encode($json_data);
        }
        
        // 非JSON或无法解析的情况，返回原始数据
        // 可以根据需要添加其他类型数据的脱敏处理
        return $data;
    }

    public function getProxySQL($cluster_id)
    {
        $sql = "select * from proxysql_nodes where db_cluster_id={$cluster_id} order by id asc limit 1";
        return $this->getList($sql);
    }


    public function getClusterMgrHttp($meta_hosts):string
    {
        $slave_meta_items = explode(",",  $meta_hosts);

        $slave_meta_items_arr = explode(":",  $slave_meta_items[0]);


        $get_mgr_nods_sql="SELECT * from cluster_mgr_nodes WHERE member_state='source'";

        $mgr_nodes = $this->getMysql($slave_meta_items_arr[0],$slave_meta_items_arr[1],'pgx','pgx_pwd','kunlun_metadata_db',$get_mgr_nods_sql);

        if ($mgr_nodes){
            $mgr_ip=$mgr_nodes[1];
            $mgr_port=$mgr_nodes[2];

            return $mgr_http="http://{$mgr_ip}:{$mgr_port}/HttpService/Emit";
        }
        return "";
    }


    //mysql连接
    public function getMysqlNullDb($host, $port, $username, $pwd, $dbname,$sql)
    {
        mysqli_report(MYSQLI_REPORT_STRICT);
        try {
            // 连接数据库服务器（不指定数据库）
            $conn = mysqli_connect("{$host}:{$port}", $username, $pwd);

            // 检查连接是否成功
            if (!$conn) {
                $arr['code'] = 500;
                $arr['message'] = 'Connection failed: ' . mysqli_connect_error();
                return $arr;
            }

            // 尝试选择数据库
            if (!mysqli_select_db($conn, $dbname)) {
                // 如果数据库不存在，则创建数据库
                $createDbSql = "CREATE DATABASE $dbname";
                if (mysqli_query($conn, $createDbSql)) {
                    $arr['message'] = 'Database created successfully';
                } else {
                    $arr['code'] = 500;
                    $arr['message'] = 'Database creation failed: ' . mysqli_error($conn);
                    mysqli_close($conn);
                    return $arr;
                }

                // 再次选择新创建的数据库
                if (!mysqli_select_db($conn, $dbname)) {
                    $arr['code'] = 500;
                    $arr['message'] = 'Database selection failed after creation: ' . mysqli_error($conn);
                    mysqli_close($conn);
                    return $arr;
                }
            }

            // 执行插入操作
            if (mysqli_query($conn, $sql)) {
                $arr['code'] = 200;
                $arr['message'] = 'Record inserted successfully';
            } else {
                $arr['code'] = 500;
                $arr['message'] = 'Insert failed: ' . mysqli_error($conn);
            }

            // 关闭连接
            mysqli_close($conn);
            return $arr;
        } catch (Exception $e) {
            // 捕获异常
            $arr['message'] = iconv('gbk', 'utf-8', $e->getMessage());
            $arr['code'] = 500;
            return $arr;
        }
    }


    //mysql连接
    public function getMysql($host, $port, $username, $pwd, $dbname, $sql)
    {
        mysqli_report(MYSQLI_REPORT_STRICT);
        try {
            $conn = mysqli_connect("{$host}:{$port}", $username, $pwd, $dbname);
            $query = mysqli_query($conn, $sql);
            $arr = mysqli_fetch_row($query);
            $list = [];
            if ($result = mysqli_query($conn, $sql)) {
                // 一条条获取
                while ($row = mysqli_fetch_row($result)) {
                    $row_num = ['TABLE_SCHEMA' => $row[0], 'TABLE_NAME' => $row[1]];
                    $list[] = $row_num;
                }
                // 释放结果集合
                mysqli_free_result($result);
            }
            $arr['code'] = 200;
            $arr['list'] = $list;
            mysqli_close($conn);
            return $arr;
        } catch (Exception $e) {
            $arr[0] = iconv('gbk', 'utf-8', $e->getMessage());
            $arr['code'] = 500;
            return $arr;
        }
    }

    public function getMysqlRes($host, $port, $username, $pwd, $dbname, $sql)
    {
        mysqli_report(MYSQLI_REPORT_STRICT);
        try {
            $conn = mysqli_connect("{$host}:{$port}", $username, $pwd, $dbname);
            // 设置连接的字符集为utf8
            mysqli_set_charset($conn, 'utf8');
            $list = [];
            if ($result = mysqli_query($conn, $sql)) {
                // 一条条获取
                while ($row = mysqli_fetch_assoc($result)) {
                    $list[] = $row; // 每行数据已为键值对
                }
                // 释放结果集合
                mysqli_free_result($result);
            }
            $arr = [
                'code' => 200,
                'list' => $list
            ];
            mysqli_close($conn);
            return $arr;
        } catch (Exception $e) {
            $arr = [
                'code' => 500,
                'message' => iconv('gbk', 'utf-8', $e->getMessage())
            ];
            return $arr;
        }
    }


    //mysql_node连接
    public function getMysqlNode($host, $port, $username, $pwd, $dbname, $sql)
    {
        mysqli_report(MYSQLI_REPORT_STRICT);
        try {
            $conn = mysqli_connect("{$host}:{$port}", $username, $pwd, $dbname);
            $query = mysqli_query($conn, $sql);
            $arr = mysqli_fetch_row($query);
            $list = [];
            if ($result = mysqli_query($conn, $sql)) {
                // 一条条获取
                while ($row = mysqli_fetch_row($result)) {
                    $row_num = ['name' => $row[0], 'value' => $row[1]];
                    $list[] = $row_num;
                }
                // 释放结果集合
                mysqli_free_result($result);
            }
            $arr['code'] = 200;
            $arr['list'] = $list;
            mysqli_close($conn);
            return $arr;
        } catch (Exception $e) {
            $arr[0] = iconv('gbk', 'utf-8', $e->getMessage());
            $arr['code'] = 500;
            return $arr;
        }
    }

    //pgsql
    public function sqlUser($sql, $host, $port, $username, $pwd)
    {
        $database = $this->pg_database;
        $conn = pg_connect("host=$host dbname=$database user=$username password=$pwd port=$port");
        @$query = pg_query($conn, $sql);
        if (!$query) {
            $data['error'] = pg_last_error($conn);
            $data['code'] = 500;
            return $data;
        } else {
            //查询语句
            //获取字段的参数名pg_field_name
            $td = [];
            $tdn = pg_num_rows($query);
            for ($i = 0; $i < $tdn; ++$i) {
                $td[] = pg_fetch_object($query, $i);
            }
            $data['list'] = $td;
            $data['code'] = 200;
            pg_close($conn);
            return $data;
        }
    }


    public function PgQuery($sql, $host, $port, $username, $pwd)
    {
        $database = $this->pg_database;
        $data = [];
        try {
            $conn = pg_connect("host=$host dbname=$database user=$username password=$pwd port=$port");
            if (!$conn) {
                throw new Exception('Unable to connect to the database.');
            }
            $result = pg_query($conn, $sql);
            if (!$result) {
                throw new Exception(pg_last_error($conn));
            }
            $data['list'] = [];
            while ($row = pg_fetch_object($result)) {
                $data['list'][] = $row;
            }
            $data['code'] = 200;
        } catch (Exception $e) {
            $data['error'] = $e->getMessage();
            $data['code'] = 500;
        } finally {
            if ($conn) {
                pg_close($conn);
            }
        }
        return $data;
    }


    public function sqlUserTotal($sql, $host, $port, $username, $pwd)
    {
        $database = $this->pg_database;
        $conn = pg_connect("host=$host dbname=$database user=$username password=$pwd port=$port");
        @$query = pg_query($conn, $sql);
        if (!$query) {
            $data['error'] = pg_last_error($conn);
            $data['code'] = 500;
            return $data;
        } else {
            $tdn = pg_fetch_row($query);
            $data['total'] = (int)$tdn[0];
            $data['code'] = 200;
            pg_close($conn);
            return $data;
        }
    }

    public function DB($sql, $host, $port, $username): array
    {
        $database = $this->pg_database;
        $pwd = $username;
        error_reporting(0);//禁止显示PHP警告提示
        $conn = pg_connect("host=$host dbname=$database user=$username password=$pwd port=$port");
        if (!$conn) {
            $data['error'] = 'pg_connect(): Unable to connect to PostgreSQL server';
            $data['code'] = 500;
            return $data;
        }

        @$query = pg_query($conn, $sql);
        $q = pg_result_status($query);
        if (!$query) {
            $data['error'] = pg_last_error($conn);
            $data['code'] = 500;
            return $data;
        } else {
            if ($q == 1) {
                $data['code'] = 200;
                $data['q'] = $q;
                return $data;
            } else {
                //查询语句
                //获取字段的参数名pg_field_name
                $td = [];
                $tdn = pg_num_rows($query);
                for ($i = 0; $i < $tdn; ++$i) {
                    $td[] = pg_fetch_object($query, $i);
                }
                $data['code'] = 200;
                $data['arr'] = $td;
                $data['q'] = $q;
                return $td;
            }
            pg_close($conn);
        }
    }

    //计时函数
    function getmicrotime()
    {
        [$usec, $sec] = explode(" ", microtime());
        return (float)sprintf('%.0f', (floatval($usec) + floatval($sec)) * 1000);
    }

    public function getResult($text, $host, $port, $username, $dbname)
    {
        //$dbname=$_SESSION['database'];
        //$username=$_SESSION['username'];
        //$host=$this->server_ip;
        //$port=$this->port;
        $pwd = $username;
        $conn = pg_connect("host=$host dbname=$dbname user=$username password=$pwd port=$port");
        //计时开始
        //exit(print_r($conn));
        $time_start = $this->getmicrotime();
        @$query = pg_query($conn, $text);
        //计时结束.
        $time_end = $this->getmicrotime();
        $times = $time_end - $time_start;
        //插入/更新语句
        $q = pg_result_status($query);
        if (!$query) {
            $data['code'] = 500;
            $data['error'] = pg_last_error($conn);
            return $data;
        } else {
            if ($q == 1) {
                $res['q'] = $q;
                $res['times'] = $times;
                $res['code'] = 200;
                return $res;
            } else {
                //查询语句
                //获取字段的参数名pg_field_name
                $num = pg_num_fields($query);
                $th = [];
                $td = [];
                for ($j = 0; $j < $num; ++$j) {
                    $colname[$j] = pg_field_name($query, $j);
                    $th[] = $colname[$j];
                }
                $tdn = pg_num_rows($query);
                for ($i = 0; $i < $tdn; ++$i) {
                    $td[] = pg_fetch_object($query, $i);
                }

                if (is_object($td)) {
                    $td = (array)$td;
                } else {
                    $td = $this->toArray($td);
                }
                $data['title'] = $th;
                $data['arr'] = $td;
                $data['times'] = $times;
                $data['q'] = $q;
                $data['code'] = 200;
                return $data;
            }
            pg_close($conn);
        }
    }

    public function toArray($td)
    {
        if (is_array($td)) {
            foreach ($td as $key => $value) {
                $td[$key] = (array)$value;
            }
        }
        return $td;
    }

    public function getClusterMgrMaster($ip, $port)
    {
        $url = 'http://' . $ip . ':' . $port . '/HttpService/Emit';
        $httpcode = 0;
        $ch = curl_init();
        $timeout = 1; // 设置超时的时间[单位：秒]
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HEADER, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_exec($ch);
        # 获取状态码赋值
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if ($httpcode == 200) {
            return true;
        } else {
            return false;
        }
    }

    public function ipTranslation($host)
    {
        if (filter_var($host, FILTER_VALIDATE_IP)) {
            $ip = $host;
        } elseif (filter_var($host, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME)) {
            if (PHP_OS == 'Linux') {
                $string = file_get_contents('./json/host.json');
                $string = json_decode($string, true);
                foreach ($string['ips'] as $k => $v) {
                    if ($v['addr'] == $host) {
                        $ip = $v['ip'];
                    }
                }
            } else {
                $ip = gethostbyname($host);
            }
            if (!isset($ip)) {
                $ip = $host;
            }
        }
        return $ip;
    }


    public function getDBName($ip, $port, $user_name): array
    {
        $sql_main = "select datname from pg_database where datname not in('template0','template1');";
        $res = $this->DB($sql_main, $ip, $port, $user_name);
        return $res;
    }


    public function getPGTableList($comp_nodes = []): array
    {
        $result = [];
        foreach ($comp_nodes as $knode => $vnode) {
            //连接该计算节点取库名
            $host = $this->ipTranslation($vnode['hostaddr']);
            $res_main = $this->getDBName($host, $vnode['port'], $vnode['user_name']);
            if ($res_main['code'] == 500) {
                return $result;
            } else {
                $ip = $vnode['hostaddr'];
                $port = $vnode['port'];
                $name = $vnode['user_name'];
                foreach ($res_main as $knode => $item) {
                    $tmp = [];
                    $tmp["ip"] = $ip;
                    $tmp["port"] = $port;
                    $tmp["user"] = $name;
                    $tmp["value"] = $item->datname;
                    $tmp["desc"] = $item->datname;
                    $tmp["label"] = $item->datname;
                    $schema = "select oid,* from pg_catalog.pg_namespace where nspname not in('pg_toast','pg_temp_1','pg_catalog','information_schema','pg_toast_temp_1');";
                    $scheam = $this->getResult($schema, $host, $port, $name, $item->datname);
                    if ($scheam["code"] == 200) {
                        $tmp["children"] = [];
                        foreach ($scheam["arr"] as $v => $s) {
                            $_schema = [];
                            $_schema["label"] = $s["nspname"];
                            $_schema["value"] = $s["nspname"];
                            $_schema["desc"] = $s["nspname"];
                            $_schema["children"] = [];

                            $sql_main = "SELECT relname FROM pg_class t1 JOIN pg_namespace t2 ON t1.relnamespace = t2.oid WHERE t2.nspname = '" . $s['nspname'] . "' AND (relkind = 'p' OR (relkind = 'r' AND relispartition = false AND relshardid != 0))";
                            
                            //SELECT relname  FROM pg_class t1 JOIN pg_namespace t2 ON t1.relnamespace = t2.oid WHERE t2.nspname = '__test_data__'   AND (       relkind = 'p'        OR (relkind = 'r' AND relispartition = false AND relshardid != 0)   );
                            
                            $res = $this->getResult($sql_main, $host, $port, $name, $item->datname);
                            if ($res["code"] == 200) {
                                foreach ($res["arr"] as $v => $val) {
                                    $_table = [];
                                    $_table["label"] = $val["relname"];
                                    $_table["value"] = $val["relname"];
                                    $_table["desc"] = $val["relname"];
                                    $_schema["children"][] = $_table;
                                }
                            }
                            $tmp["children"][] = $_schema;
                        }
                    }
                    $result[] = $tmp;
                }
            }
        }
        return $result;
    }

    /**
     * 获取告警记录总数
     * @param array $where 查询条件
     * @return int 记录总数
     */
    public function getAlarmRecordCount($where = [])
    {
        $this->db->from('cluster_alarm_info');
        
        if (!empty($where)) {
            foreach ($where as $key => $value) {
                $this->db->where($key, $value);
            }
        }
        
        return $this->db->count_all_results();
    }

    /**
     * 获取告警记录列表
     * @param array $where 查询条件
     * @param int $offset 偏移量
     * @param int $limit 每页记录数
     * @return array 告警记录列表
     */
    public function getAlarmRecordList($where = [], $offset = 0, $limit = 10)
    {
        $this->db->select('*');
        $this->db->from('cluster_alarm_info');
        
        if (!empty($where)) {
            foreach ($where as $key => $value) {
                $this->db->where($key, $value);
            }
        }
        
        $this->db->order_by('occur_timestamp', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result_array();
    }
}
