<?php

class Db_model extends CI_Model
{
    protected $db;
    protected $config;
    protected $pg_database;
    protected $pg_username;
    protected $default_username;
    protected $post_url;
    protected $grafana_key;
    protected $dbRole;

    public function __construct()
    {
        parent::__construct();
        $this->db = $this->load->database('default', true);
        $this->dbRole = $this->load->database('role', true);
        
        // 加载配置文件
        $this->config = $this->load->config('myconfig', true);
        
        // 设置默认值，避免配置文件不存在时的错误
        $this->pg_database = $this->config['pg_database'] ?? '';
        $this->pg_username = $this->config['pg_username'] ?? '';
        $this->default_username = $this->config['default_username'] ?? '';
        $this->post_url = $this->config['post_url'] ?? '';
        $this->grafana_key = $this->config['grafana_key'] ?? '';
    }

    // 获取 cluster_steam_task 表中的记录


    public function CreateTable()
    {
        try {
            // 告警记录表
            $sql_table = "CREATE TABLE IF NOT EXISTS `cluster_alarm_info` (
                `id` bigint unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `job_id` VARCHAR(128) DEFAULT NULL COMMENT '任务ID',
                `alarm_level` enum('FATAL','ERROR','WARNING') NOT NULL DEFAULT 'WARNING' COMMENT '告警级别:FATAL-致命错误,ERROR-严重错误,WARNING-警告信息',
                `alarm_type` varchar(128) DEFAULT NULL COMMENT '告警类型',
                `status` enum('unhandled','handled','ignore') NOT NULL DEFAULT 'unhandled' COMMENT '告警状态:unhandled-未处理,send-推送告警,handled-已处理,ignore-已忽略',
                `occur_timestamp` timestamp(6) NOT NULL DEFAULT CURRENT_TIMESTAMP(6) COMMENT '告警发生时间',
                `job_info` text COMMENT '告警详细信息(JSON格式)',
                `handle_time` timestamp(6) NULL DEFAULT NULL COMMENT '处理时间',
                `handler_name` varchar(128) DEFAULT NULL COMMENT '处理人',
                `cluster_id` int unsigned DEFAULT NULL COMMENT '集群ID',
                `shardid` int unsigned DEFAULT NULL COMMENT '分片ID',
                `compnodeid` int unsigned DEFAULT NULL COMMENT '计算节点ID',
                `svrid` int unsigned DEFAULT NULL COMMENT '服务器ID',
                PRIMARY KEY (`id`),
                KEY `idx_alarm_type` (`alarm_type`),
                KEY `idx_status` (`status`),
                KEY `idx_occur_timestamp` (`occur_timestamp`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警记录表';";
            $this->updateList($sql_table);

            // 告警用户配置表 - 使用tableExists检查表是否存在
            if (!$this->tableExists('kunlun_metadata_db', 'cluster_alarm_user')) {
                $sql_table = "CREATE TABLE `kunlun_metadata_db`.`cluster_alarm_user` (
                    `id` varchar(128) NOT NULL COMMENT '告警类型ID',
                    `label` varchar(255) DEFAULT NULL COMMENT '告警类型名称',
                    `accept_user` varchar(255) DEFAULT NULL COMMENT '接收用户列表',
                    `create_at` int DEFAULT NULL COMMENT '创建时间',
                    `push_type` varchar(255) DEFAULT NULL COMMENT '推送方式:system-系统提醒,phone_message-短信,mail-邮件,wechat-微信',
                    `threshold` tinyint DEFAULT NULL COMMENT '告警阈值',
                    `level` varchar(255) DEFAULT NULL COMMENT '告警级别',
                    `status` enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用',
                    PRIMARY KEY (`id`),
                    KEY `idx_status` (`status`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警用户配置表';";
                $this->db->query($sql_table);
                log_message('info', '创建表 cluster_alarm_user 成功');
            }

            // 告警推送配置表
            $push_config = "CREATE TABLE IF NOT EXISTS `kunlun_metadata_db`.`cluster_alarm_message_config` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `message` text COMMENT '推送配置信息(JSON格式)',
                `create_at` INT DEFAULT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `type` VARCHAR(64) DEFAULT NULL COMMENT '推送类型:email-邮件,phone_message-短信,wechat-微信',
                `status` enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用',
                PRIMARY KEY (`id`),
                UNIQUE KEY `uk_type` (`type`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警推送配置表';";
            $this->updateList($push_config);

            // 告警推送日志表
            $push_log = "CREATE TABLE IF NOT EXISTS `kunlun_metadata_db`.`cluster_alarm_push_log` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `alarm_type` VARCHAR(128) DEFAULT NULL COMMENT '告警类型',
                `push_type` VARCHAR(128) DEFAULT NULL COMMENT '推送类型',
                `content` text COMMENT '推送内容',
                `content_res` VARCHAR(255) DEFAULT NULL COMMENT '推送结果',
                `create_at` INT DEFAULT NULL COMMENT '创建时间',
                `status` enum('success','failed') NOT NULL DEFAULT 'success' COMMENT '推送状态',
                `error_msg` text DEFAULT NULL COMMENT '错误信息',
                PRIMARY KEY (`id`),
                KEY `idx_alarm_type` (`alarm_type`),
                KEY `idx_push_type` (`push_type`),
                KEY `idx_create_at` (`create_at`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警推送日志表';";
            $this->updateList($push_log);

            // 告警代理服务器配置表
            $proxy_config = "CREATE TABLE IF NOT EXISTS `kunlun_metadata_db`.`cluster_alarm_proxy_config` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `server_address` VARCHAR(255) NOT NULL COMMENT '服务器地址',
                `template_id` VARCHAR(255) DEFAULT NULL COMMENT '模板ID',
                `create_at` INT NOT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `status` enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用',
                `last_check_time` INT DEFAULT NULL COMMENT '最后检测时间',
                `last_check_status` enum('success','failed') DEFAULT NULL COMMENT '最后检测状态',
                PRIMARY KEY (`id`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警代理服务器配置表';";
            $this->updateList($proxy_config);

            // 告警配置表
            $alarm_config = "CREATE TABLE IF NOT EXISTS `kunlun_metadata_db`.`kunlun_alarm_config` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `config_type` VARCHAR(64) NOT NULL COMMENT '配置类型:email,sms,wechat,proxy',
                `config_data` TEXT NOT NULL COMMENT '配置数据(JSON格式)',
                `create_at` INT NOT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `status` enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用',
                PRIMARY KEY (`id`),
                UNIQUE KEY `uk_type` (`config_type`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='告警配置表';";
            $this->updateList($alarm_config);

            // 告警推送历史记录表
            $history_alarm_push = "CREATE TABLE IF NOT EXISTS `kunlun_metadata_db`.`cluster_alarm_push_history` (
                `id` int(11) NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `alarm_id` int(11) NOT NULL COMMENT '告警ID',
                `push_type` varchar(20) NOT NULL COMMENT '推送类型：phone_message, mail, wechat, proxy',
                `receiver` varchar(255) NOT NULL COMMENT '接收者',
                `content` text NOT NULL COMMENT '推送内容',
                `status` varchar(20) NOT NULL DEFAULT 'pending' COMMENT '推送状态：pending, success, failed',
                `push_time` datetime NOT NULL COMMENT '推送时间',
                `error_message` text COMMENT '错误信息',
                `is_manual` tinyint(1) NOT NULL DEFAULT '0' COMMENT '是否手动推送：0-自动，1-手动',
                PRIMARY KEY (`id`),
                KEY `idx_alarm_id` (`alarm_id`),
                KEY `idx_push_time` (`push_time`),
                KEY `idx_status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT='告警推送历史记录表';";
            $this->updateList($history_alarm_push);

           
            $sql = "CREATE TABLE IF NOT EXISTS  auto_customer_service (
                id int unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                hostaddr varchar(8192) CHARACTER SET latin1 COLLATE latin1_swedish_ci NOT NULL COMMENT '主机地址',
                port int unsigned DEFAULT NULL COMMENT '端口号',
                PRIMARY KEY (id)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='自动客户服务表';";
            $this->updateList($sql);


            $sql = "CREATE TABLE IF NOT EXISTS kunlun_dba_tools_db.shard_max_dalay (
                id int unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                shard_id int unsigned NOT NULL COMMENT '分片ID',
                db_cluster_id int unsigned NOT NULL COMMENT '集群ID',
                user_id int unsigned NOT NULL COMMENT '用户ID',
                max_delay_time int NOT NULL DEFAULT '100' COMMENT '最大延迟时间(秒)',
                when_created timestamp NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
                update_time timestamp NULL DEFAULT CURRENT_TIMESTAMP COMMENT '更新时间',
                PRIMARY KEY (id)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='分片最大延迟配置表';";
            $this->updateList($sql);


            $sql_table = "CREATE TABLE IF NOT EXISTS `cluster_cdc_server` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `host_addr` VARCHAR(128) CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC服务器地址',
                `port` VARCHAR(8) CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC服务器端口',
                `master` INT NOT NULL COMMENT '是否为主节点：1-是，0-否',
                `create_time` INT NOT NULL COMMENT '创建时间',
                `status` INT NOT NULL COMMENT '状态：1-正常，0-异常',
                `group_name` VARCHAR(255) COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC集群组名称',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB AUTO_INCREMENT=12 DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC服务器节点表';";
            $this->updateList($sql_table);


            $cdc_worker_table = "CREATE TABLE IF NOT EXISTS `cluster_cdc_worker` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `worker_param` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC工作参数(JSON格式)',
                `create_time` INT NOT NULL COMMENT '创建时间',
                `job_id` INT NOT NULL COMMENT '任务ID',
                `status` VARCHAR(128) CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT '状态',
                `worker_status` VARCHAR(255) COLLATE utf8mb4_0900_as_cs DEFAULT NULL COMMENT '工作状态',
                `group_name` VARCHAR(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC集群组名称',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB AUTO_INCREMENT=96 DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC工作任务表';";
            $this->updateList($cdc_worker_table);


            $cdc_job_state = "CREATE TABLE IF NOT EXISTS `cdc_job_state` (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `param` TEXT COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT '任务参数(JSON格式)',
                `job_id` INT NOT NULL COMMENT '任务ID',
                `group_name` VARCHAR(255) COLLATE utf8mb4_0900_as_cs NOT NULL COMMENT 'CDC集群组名称',
                `time` INT DEFAULT NULL COMMENT '记录时间',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC任务状态表';";
            $this->updateList($cdc_job_state);

            $create_sql = "CREATE TABLE IF NOT EXISTS `cluster_cdc_hot_switch` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `cdc_job_id` INT NOT NULL COMMENT 'CDC任务ID',
                `status` INT NOT NULL COMMENT '状态：1-成功，0-失败',
                `create_at` INT NOT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `delete_at` INT DEFAULT NULL COMMENT '删除时间',
                `switch_params` TEXT DEFAULT NULL COMMENT '切换参数(JSON格式)',
                `switch_success` INT NOT NULL COMMENT '切换是否成功：1-成功，0-失败',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC主备切换记录表';";
            $this->updateList($create_sql);


            $sql = "CREATE TABLE IF NOT EXISTS `cluster_type_info` (
                `id` int unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `db_cluster_id` int unsigned NOT NULL COMMENT '集群ID',
                `cluster_type` enum('local','idc') DEFAULT NULL COMMENT '集群类型：local-本地，idc-数据中心',
                `idcs` text COMMENT '数据中心信息',
                `shard_ids` varchar(64) DEFAULT NULL COMMENT '分片ID列表',
                `dc_type` enum('same_city','cross_city') DEFAULT NULL COMMENT '数据中心类型：same_city-同城，cross_city-异地',
                `status` enum('inuse','deleted') DEFAULT 'inuse' COMMENT '状态：inuse-使用中，deleted-已删除',
                `user_name` varchar(128) DEFAULT NULL COMMENT '用户名',
                PRIMARY KEY (`id`),
                UNIQUE KEY `name` (`db_cluster_id`)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='集群类型信息表';";
            $this->upRoledateList($sql);

            $sql = "CREATE TABLE IF NOT EXISTS `variables_record` (
                `id` bigint unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `variables` text COMMENT '变量信息',
                `when_started` timestamp(6) NOT NULL DEFAULT CURRENT_TIMESTAMP(6) COMMENT '开始时间',
                `node_type` enum('storage','computer') NOT NULL DEFAULT 'computer' COMMENT '节点类型：storage-存储节点，computer-计算节点',
                `user_name` varchar(128) DEFAULT NULL COMMENT '用户名',
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='变量记录表';";
            $this->updateList($sql);


            $create_sql = "CREATE TABLE IF NOT EXISTS `cluster_steams_recover` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `stream_id` INT NOT NULL COMMENT '流ID',
                `status` INT NOT NULL COMMENT '状态：1-成功，0-失败',
                `create_at` INT NOT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `delete_at` INT DEFAULT NULL COMMENT '删除时间',
                `prv_params` TEXT DEFAULT NULL COMMENT '原参数(JSON格式)',
                `now_params` TEXT DEFAULT NULL COMMENT '当前参数(JSON格式)',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC流恢复记录表';";
            $this->updateList($create_sql);


            $create_sql = "CREATE TABLE IF NOT EXISTS `cluster_steams` (
                `id` INT NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `rcr_id` INT NOT NULL COMMENT 'RCR ID',
                `cdc_src` INT NOT NULL COMMENT 'CDC源端ID',
                `cdc_dst` INT NOT NULL COMMENT 'CDC目标端ID',
                `status` INT NOT NULL COMMENT '状态：1-正常，0-异常',
                `create_at` INT NOT NULL COMMENT '创建时间',
                `update_at` INT DEFAULT NULL COMMENT '更新时间',
                `delete_at` INT DEFAULT NULL COMMENT '删除时间',
                `task_id` VARCHAR(128) COLLATE utf8mb4_0900_as_cs DEFAULT NULL COMMENT '任务ID',
                `task_name` VARCHAR(128) COLLATE utf8mb4_0900_as_cs DEFAULT NULL COMMENT '任务名称',
                `task_params` TEXT DEFAULT NULL COMMENT '任务参数(JSON格式)',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='CDC流任务表';";
            $this->updateList($create_sql);


            $sql = "CREATE TABLE IF NOT EXISTS `cluster_job_record` (
                `id` bigint unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `job_id` varchar(128) DEFAULT NULL COMMENT '任务ID',
                `sync_num` smallint unsigned NOT NULL DEFAULT '1' COMMENT '同步次数',
                `job_type` varchar(128) DEFAULT NULL COMMENT '任务类型',
                `status` enum('not_started','ongoing','done','failed') NOT NULL DEFAULT 'not_started' COMMENT '状态：not_started-未开始，ongoing-进行中，done-已完成，failed-失败',
                `memo` text COMMENT '备注信息',
                `job_info` text COMMENT '任务信息(JSON格式)',
                `when_started` timestamp(6) NOT NULL DEFAULT CURRENT_TIMESTAMP(6) COMMENT '开始时间',
                `when_ended` timestamp(6) NULL DEFAULT NULL COMMENT '结束时间',
                `user_name` varchar(128) DEFAULT NULL COMMENT '用户名',
                PRIMARY KEY (`id`),
                UNIQUE KEY `id` (`id`)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='集群任务记录表';";
            $this->updateList($sql);


            $sql = "CREATE TABLE IF NOT EXISTS rcr_max_dalay (
                id int unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                rcr_id int unsigned NOT NULL COMMENT 'RCR ID',
                user_id int unsigned NOT NULL COMMENT '用户ID',
                max_delay_time int NOT NULL DEFAULT '0' COMMENT '最大延迟时间(秒)',
                when_created timestamp NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
                update_time timestamp NULL DEFAULT CURRENT_TIMESTAMP COMMENT '更新时间',
                PRIMARY KEY (id)
            ) ENGINE=InnoDB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='RCR最大延迟配置表';";
            $this->upRoledateList($sql);

            // 添加性能指标表
            $performance_metrics = "CREATE TABLE IF NOT EXISTS `cluster_performance_metrics` (
                `id` bigint unsigned NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `metric_type` varchar(64) NOT NULL COMMENT '指标类型:alarm_processing-告警处理,push_performance-推送性能,system_resources-系统资源',
                `metric_data` text NOT NULL COMMENT '指标数据(JSON格式)',
                `create_at` int NOT NULL COMMENT '创建时间',
                PRIMARY KEY (`id`),
                KEY `idx_metric_type` (`metric_type`),
                KEY `idx_create_at` (`create_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_0900_as_cs COMMENT='性能指标表';";
            $this->updateList($performance_metrics);


            $sql = "CREATE TABLE IF NOT EXISTS kunlun_dba_tools_db.cluster_meta_info (
                `id` INT UNSIGNED NOT NULL AUTO_INCREMENT COMMENT '主键ID',
                `user_id` INT UNSIGNED NOT NULL COMMENT '用户ID',
                `name` VARCHAR(120) DEFAULT NULL COMMENT '集群名称',
                `rcr_meta` VARCHAR(255) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT 'RCR元数据',
                `db_type` VARCHAR(64) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库类型',
                `db_user` VARCHAR(64) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库用户名',
                `db_passwd` VARCHAR(128) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库密码',
                `db_name` VARCHAR(64) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库名称',
                `when_created` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
                `update_time` TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP COMMENT '更新时间',
                PRIMARY KEY (`id`)
            ) ENGINE=INNODB AUTO_INCREMENT=1 DEFAULT CHARSET=utf8mb3 COMMENT='集群元数据信息表';";
            $this->updateList($sql);
            


            log_message('info', '数据库表创建完成');
        } catch (Exception $e) {
            log_message('error', '数据库表创建失败: ' . $e->getMessage());
        }
    }

    /**
     * 检查表是否存在
     * @param string $schema 数据库名
     * @param string $table 表名
     * @return bool 表是否存在
     */
    private function tableExists($schema, $table)
    {
        try {
            $sql = "SELECT COUNT(*) as count FROM information_schema.TABLES 
                    WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?";
            $result = $this->getList($sql, [$schema, $table]);
            
            return $result && $result[0]['count'] > 0;
        } catch (Exception $e) {
            log_message('error', "检查表是否存在失败 $schema.$table: " . $e->getMessage());
            return false;
        }
    }

    public function AlterTable()
    {
        try {
            // 为告警记录表添加索引
            // 检查索引是否存在，不存在则添加
            $this->addIndexIfNotExists('cluster_alarm_info', 'idx_alarm_type', 'alarm_type');
            $this->addIndexIfNotExists('cluster_alarm_info', 'idx_status', 'status');
            $this->addIndexIfNotExists('cluster_alarm_info', 'idx_occur_timestamp', 'occur_timestamp');
            
            // 为告警推送配置表添加状态字段
            $this->addColumnIfNotExists('cluster_alarm_message_config', 'status', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'type');
            $this->addIndexIfNotExists('cluster_alarm_message_config', 'idx_status', 'status');
            
            // 为告警推送日志表添加状态和错误信息字段
            $this->addColumnIfNotExists('cluster_alarm_push_log', 'status', "enum('success','failed') NOT NULL DEFAULT 'success' COMMENT '推送状态'", 'create_at');
            $this->addColumnIfNotExists('cluster_alarm_push_log', 'error_msg', "text DEFAULT NULL COMMENT '错误信息'", 'status');
            $this->addIndexIfNotExists('cluster_alarm_push_log', 'idx_status', 'status');
            
            // 为告警代理配置表添加检测状态字段
            $this->addColumnIfNotExists('kunlun_metadata_db.cluster_alarm_proxy_config', 'last_check_time', "INT DEFAULT NULL COMMENT '最后检测时间'", 'status');
            $this->addColumnIfNotExists('kunlun_metadata_db.cluster_alarm_proxy_config', 'last_check_status', "enum('success','failed') DEFAULT NULL COMMENT '最后检测状态'", 'last_check_time');
            
            // CDC Worker相关字段
            $this->addColumnIfNotExists('cluster_cdc_worker', 'delete_time', "INT COMMENT '删除时间'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'db_cluster_id', "INT COMMENT '数据库集群ID'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'is_proxy_sql', "int NULL DEFAULT 0 COMMENT '是否代理SQL：0-否，1-是'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'job_name', "VARCHAR(128) COMMENT '任务名称'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'src_db_type', "VARCHAR(64) COMMENT '源数据库类型'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'dst_db_type', "VARCHAR(64) COMMENT '目标数据库类型'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'error_message', "text COMMENT '错误信息'");
            $this->addColumnIfNotExists('cluster_cdc_worker', 'last_sync_time', "int NULL DEFAULT 0 COMMENT '最后同步时间'");

            // 为CDC流任务表添加状态字段
            $this->addColumnIfNotExists('cluster_steams', 'status_enum', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'delete_at');
            $this->addIndexIfNotExists('cluster_steams', 'idx_status_enum', 'status_enum');
            
            // 为CDC流恢复记录表添加状态字段
            $this->addColumnIfNotExists('cluster_steams_recover', 'status_enum', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'delete_at');
            $this->addIndexIfNotExists('cluster_steams_recover', 'idx_status_enum', 'status_enum');
            
            // 为CDC主备切换记录表添加状态字段
            $this->addColumnIfNotExists('cluster_cdc_hot_switch', 'status_enum', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'delete_at');
            $this->addIndexIfNotExists('cluster_cdc_hot_switch', 'idx_status_enum', 'status_enum');
            
            // 为CDC服务器节点表添加状态字段
            $this->addColumnIfNotExists('cluster_cdc_server', 'status_enum', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'status');
            $this->addIndexIfNotExists('cluster_cdc_server', 'idx_status_enum', 'status_enum');
            
            // 为集群任务记录表添加状态字段
            $this->addColumnIfNotExists('cluster_job_record', 'status_enum', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'status');
            $this->addIndexIfNotExists('cluster_job_record', 'idx_status_enum', 'status_enum');
            
            // 为CDC任务状态表添加状态字段
            $this->addColumnIfNotExists('cdc_job_state', 'status', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'time');
            $this->addIndexIfNotExists('cdc_job_state', 'idx_status', 'status');
            
            // 为告警用户配置表添加状态字段
            $this->addColumnIfNotExists('kunlun_metadata_db.cluster_alarm_user', 'status', "enum('enabled','disabled') NOT NULL DEFAULT 'enabled' COMMENT '是否启用'", 'push_type');
            $this->addIndexIfNotExists('kunlun_metadata_db.cluster_alarm_user', 'idx_status', 'status');

            // Unknown column 'db_user' in 'field list' - Invalid query: select `db_type` ,`db_user`,`db_passwd`,`db_name` from cluster_meta_info 
            $this->addColumnIfNotExists('kunlun_dba_tools_db.cluster_meta_info', 'db_user', "VARCHAR(64) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库用户名'");
            $this->addColumnIfNotExists('kunlun_dba_tools_db.cluster_meta_info', 'db_passwd', "VARCHAR(128) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库密码'");
            $this->addColumnIfNotExists('kunlun_dba_tools_db.cluster_meta_info', 'db_name', "VARCHAR(64) COLLATE utf8mb4_0900_as_cs DEFAULT '' COMMENT '数据库名称'");
            
            log_message('info', '数据库表结构更新完成');
        } catch (Exception $e) {
            log_message('error', '数据库表结构更新失败: ' . $e->getMessage());
        }
    }
    
    /**
     * 检查并添加列
     * @param string $table 表名
     * @param string $column 列名
     * @param string $definition 列定义
     * @param string $after 在哪个列之后添加，默认为空
     * @return bool 是否成功
     */
    private function addColumnIfNotExists($table, $column, $definition, $after = '')
    {
        try {
            // 检查列是否存在
            $sql = "SELECT COUNT(*) as count FROM information_schema.COLUMNS 
                    WHERE TABLE_SCHEMA = ? 
                    AND TABLE_NAME = ? 
                    AND COLUMN_NAME = ?";
            
            // 处理带有数据库名称的表
            $tableParts = explode('.', $table);
            $params = [];
            
            if (count($tableParts) > 1) {
                $params = [$tableParts[0], $tableParts[1], $column];
            } else {
                $params = [$this->db->database, $table, $column];
            }
            
            $result = $this->getList($sql, $params);
            
            if ($result[0]['count'] == 0) {
                // 列不存在，添加列
                $afterClause = $after ? " AFTER `$after`" : "";
                
                if (count($tableParts) > 1) {
                    $sql = "ALTER TABLE `{$tableParts[0]}`.`{$tableParts[1]}` ADD COLUMN `$column` $definition$afterClause";
                } else {
                    $sql = "ALTER TABLE `$table` ADD COLUMN `$column` $definition$afterClause";
                }
                
                $this->db->query($sql);
                log_message('info', "已添加列 $column 到表 $table");
            }
            
            return true;
        } catch (Exception $e) {
            log_message('error', "添加列失败 $table.$column: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 检查并添加索引
     * @param string $table 表名
     * @param string $indexName 索引名
     * @param string $columns 列名，多列用逗号分隔
     * @return bool 是否成功
     */
    private function addIndexIfNotExists($table, $indexName, $columns)
    {
        try {
            // 检查索引是否存在
            $sql = "SELECT COUNT(*) as count FROM information_schema.STATISTICS 
                    WHERE TABLE_SCHEMA = ? 
                    AND TABLE_NAME = ? 
                    AND INDEX_NAME = ?";
            
            // 处理带有数据库名称的表
            $tableParts = explode('.', $table);
            $params = [];
            
            if (count($tableParts) > 1) {
                $params = [$tableParts[0], $tableParts[1], $indexName];
            } else {
                $params = [$this->db->database, $table, $indexName];
            }
            
            $result = $this->getList($sql, $params);
            
            if ($result[0]['count'] == 0) {
                // 索引不存在，添加索引
                if (count($tableParts) > 1) {
                    $sql = "ALTER TABLE `{$tableParts[0]}`.`{$tableParts[1]}` ADD INDEX `$indexName` ($columns)";
                } else {
                    $sql = "ALTER TABLE `$table` ADD INDEX `$indexName` ($columns)";
                }
                
                $this->db->query($sql);
                log_message('info', "已添加索引 $indexName 到表 $table");
            }
            
            return true;
        } catch (Exception $e) {
            log_message('error', "添加索引失败 $table.$indexName: " . $e->getMessage());
            return false;
        }
    }

    //查询数据
    public function getList($sql, $params = [])
    {
        try {
            // 执行带参数的 SQL 语句
            $q = $this->db->query($sql, $params);
            if ($q && $q->num_rows() > 0) {
                return $q->result_array();
            } else {
                return false;
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }


    public function upRoledateList($sql, $params = [])
    {
        try {
            // 执行带参数的 SQL 语句
            $query = $this->dbRole->query($sql, $params);
            if ($query === false) {
                // 记录错误并返回错误信息
                return $this->dbRole->error();
            }
            return $this->dbRole->affected_rows();
        } catch (Exception $e) {
            // 处理异常并返回错误代码
            return [
                'code' => 500,
                'error' => $e->getMessage()
            ];
        }
    }


    public function updateList($sql, $params = [])
    {
        try {
            // 执行带参数的 SQL 语句
            $query = $this->db->query($sql, $params);
            if ($query === false) {
                // 记录错误并返回错误信息
                return $this->db->error();
            }
            return $this->db->affected_rows();
        } catch (Exception $e) {
            // 处理异常并返回错误代码
            return [
                'code' => 500,
                'error' => $e->getMessage()
            ];
        }
    }

}
