# 日志系统优化说明

## 主要改进

1. **结构化日志**：使用JSON格式记录日志，便于分析和处理
2. **日志级别**：支持多个日志级别（Debug、Info、Warn、Error、Fatal）
3. **日志轮转**：自动轮转日志文件，防止单个文件过大
4. **上下文信息**：添加时间戳和其他上下文信息到每条日志
5. **日志管理工具**：提供日志查看、过滤和管理的脚本工具

## 日志级别说明

- **Debug**：详细的调试信息，用于开发和故障排查
- **Info**：常规操作信息，记录系统正常运行时的状态
- **Warn**：警告信息，表示可能的问题但不影响系统继续运行
- **Error**：错误信息，表示发生了影响功能的错误
- **Fatal**：致命错误，记录后会导致程序终止

## 日志配置

日志配置在 `logger/logger.go` 文件中：

```go
// 使用lumberjack进行日志轮转
rotateLogger := &lumberjack.Logger{
    Filename:   logFilePath,
    MaxSize:    100, // 每个日志文件最大100MB
    MaxBackups: 10,  // 保留10个备份
    MaxAge:     30,  // 保留30天
    Compress:   true, // 压缩旧日志
}
```

可以根据实际需求调整以下参数：

- **MaxSize**：单个日志文件的最大大小（MB）
- **MaxBackups**：保留的旧日志文件数量
- **MaxAge**：保留日志的最大天数
- **Compress**：是否压缩旧日志文件

## 日志管理工具使用方法

项目提供了 `logctl.sh` 脚本用于日志管理：

```bash
# 查看所有日志
./logctl.sh view

# 只查看错误级别的日志
./logctl.sh view error

# 实时查看最新日志
./logctl.sh tail

# 实时查看最新50行日志
./logctl.sh tail 50

# 搜索包含特定关键词的日志
./logctl.sh grep "MongoDB"

# 手动触发日志轮转
./logctl.sh rotate

# 清理30天前的日志
./logctl.sh clean 30
```

## 在代码中使用日志

```go
// 导入日志包
import "consistent_sql/logger"

// 基本用法
logger.Debug("这是一条调试信息: %s", detailInfo)
logger.Info("这是一条信息: %s", someInfo)
logger.Warn("这是一条警告: %s", warning)
logger.Error("这是一条错误: %s", err.Error())
logger.Fatal("这是一条致命错误: %s", fatalErr.Error())

// 带字段的结构化日志
logger.WithField("task_id", taskID).Info("任务开始执行")
logger.WithFields(logrus.Fields{
    "task_id": taskID,
    "user_id": userID,
    "action":  "delete",
}).Info("用户执行了操作")
```

## 性能考虑

1. **异步日志**：高负载系统可考虑使用异步日志写入方式
2. **日志级别控制**：生产环境建议使用Info级别或以上
3. **选择性日志**：对于高频操作，避免过多Debug级别日志
4. **日志大小监控**：定期检查日志大小和增长速率

## 最佳实践

1. **一致性**：在类似场景下使用一致的日志级别
2. **上下文信息**：在日志中包含相关的上下文信息（如任务ID、用户ID等）
3. **异常处理**：所有捕获的异常都应该记录日志
4. **避免敏感信息**：不要在日志中包含密码、令牌等敏感信息
5. **定期维护**：定期清理旧日志，防止磁盘空间不足 