package main

import (
	"context"
	"fmt"
	"log"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// MongoDB 诊断工具，检查 Change Stream 相关配置
func diagnoseMongoDB() {
	// MongoDB 连接配置 - 请根据实际情况修改
	mongoHost := "localhost:27017"
	username := "your_username" // 请修改为实际用户名
	password := "your_password" // 请修改为实际密码
	authDatabase := "admin"     // 认证数据库，通常是 admin

	// 构建带认证的 MongoDB URL
	var mongoURL string
	if username != "" && password != "" {
		mongoURL = fmt.Sprintf("mongodb://%s:%s@%s/%s", username, password, mongoHost, authDatabase)
	} else {
		mongoURL = fmt.Sprintf("mongodb://%s", mongoHost)
	}

	fmt.Println("=== MongoDB Change Stream 诊断工具 ===")
	fmt.Printf("连接主机: %s\n", mongoHost)
	fmt.Printf("用户名: %s\n", username)
	fmt.Printf("认证数据库: %s\n", authDatabase)

	// 连接 MongoDB
	ctx := context.Background()
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongoURL))
	if err != nil {
		log.Fatalf("连接 MongoDB 失败: %v", err)
	}
	defer client.Disconnect(ctx)

	// 测试连接
	err = client.Ping(ctx, nil)
	if err != nil {
		log.Fatalf("MongoDB ping 失败: %v", err)
	}
	fmt.Println("✓ MongoDB 连接成功")

	// 1. 检查 MongoDB 版本
	fmt.Println("\n=== 检查 MongoDB 版本 ===")
	result := client.Database("admin").RunCommand(ctx, bson.D{{"buildInfo", 1}})
	var buildInfo bson.M
	if err := result.Decode(&buildInfo); err != nil {
		fmt.Printf("❌ 获取版本信息失败: %v\n", err)
	} else {
		if version, ok := buildInfo["version"]; ok {
			fmt.Printf("✓ MongoDB 版本: %v\n", version)
		}
	}

	// 2. 检查副本集状态
	fmt.Println("\n=== 检查副本集状态 ===")
	result = client.Database("admin").RunCommand(ctx, bson.D{{"isMaster", 1}})
	var isMasterResult bson.M
	if err := result.Decode(&isMasterResult); err != nil {
		fmt.Printf("❌ 检查副本集状态失败: %v\n", err)
	} else {
		if setName, exists := isMasterResult["setName"]; exists {
			fmt.Printf("✓ 副本集名称: %v\n", setName)
			if isPrimary, ok := isMasterResult["ismaster"].(bool); ok && isPrimary {
				fmt.Println("✓ 当前节点是主节点")
			} else {
				fmt.Println("⚠️  当前节点不是主节点")
			}
		} else {
			fmt.Println("❌ 未检测到副本集配置")
			fmt.Println("   Change Stream 需要副本集模式才能工作")
			fmt.Println("   请使用以下命令启动副本集:")
			fmt.Println("   mongod --replSet rs0 --port 27017 --dbpath /data/db")
			fmt.Println("   然后在 mongo shell 中执行: rs.initiate()")
		}
	}

	// 3. 检查 oplog 状态
	fmt.Println("\n=== 检查 Oplog 状态 ===")
	oplogCollection := client.Database("local").Collection("oplog.rs")
	oplogCount, err := oplogCollection.CountDocuments(ctx, bson.M{})
	if err != nil {
		fmt.Printf("❌ 检查 oplog 失败: %v\n", err)
	} else if oplogCount > 0 {
		fmt.Printf("✓ Oplog 可用，包含 %d 条记录\n", oplogCount)

		// 获取最新的 oplog 条目
		cursor, err := oplogCollection.Find(ctx, bson.M{}, options.Find().SetSort(bson.D{{"ts", -1}}).SetLimit(1))
		if err == nil {
			defer cursor.Close(ctx)
			if cursor.Next(ctx) {
				var latestOplog bson.M
				if err := cursor.Decode(&latestOplog); err == nil {
					fmt.Printf("✓ 最新 oplog 时间戳: %v\n", latestOplog["ts"])
				}
			}
		}
	} else {
		fmt.Println("❌ Oplog 不可用或为空")
	}

	// 4. 检查用户权限
	fmt.Println("\n=== 检查用户权限 ===")
	result = client.Database("admin").RunCommand(ctx, bson.D{{"connectionStatus", 1}})
	var connStatus bson.M
	if err := result.Decode(&connStatus); err != nil {
		fmt.Printf("❌ 检查连接状态失败: %v\n", err)
	} else {
		if authInfo, ok := connStatus["authInfo"].(bson.M); ok {
			if authenticatedUsers, ok := authInfo["authenticatedUsers"].(bson.A); ok && len(authenticatedUsers) > 0 {
				fmt.Printf("✓ 已认证用户: %v\n", authenticatedUsers)
			} else {
				fmt.Println("⚠️  未检测到已认证用户（可能使用匿名连接）")
			}
		}
	}

	// 5. 测试 Change Stream 创建
	fmt.Println("\n=== 测试 Change Stream 创建 ===")
	testDB := client.Database("test_changestream")
	testCollection := testDB.Collection("test")

	// 尝试创建 Change Stream
	changeStreamOptions := options.ChangeStream()
	changeStreamOptions.SetFullDocument(options.UpdateLookup)

	pipeline := mongo.Pipeline{
		{{
			"$match", bson.D{{
				"operationType", bson.D{{
					"$in", bson.A{"insert", "update", "delete"},
				}},
			}},
		}},
	}

	changeStream, err := testCollection.Watch(ctx, pipeline, changeStreamOptions)
	if err != nil {
		fmt.Printf("❌ 创建 Change Stream 失败: %v\n", err)
		fmt.Println("   可能的原因:")
		fmt.Println("   1. MongoDB 版本过低（需要 3.6+）")
		fmt.Println("   2. 未运行在副本集模式")
		fmt.Println("   3. 权限不足")
	} else {
		fmt.Println("✓ Change Stream 创建成功")
		changeStream.Close(ctx)
	}

	// 6. 检查服务器参数
	fmt.Println("\n=== 检查服务器参数 ===")
	result = client.Database("admin").RunCommand(ctx, bson.D{{"getParameter", "*"}})
	var params bson.M
	if err := result.Decode(&params); err != nil {
		fmt.Printf("❌ 获取服务器参数失败: %v\n", err)
	} else {
		// 检查一些关键参数
		if changeStreamPreAndPostImages, ok := params["changeStreamPreAndPostImages"]; ok {
			fmt.Printf("✓ changeStreamPreAndPostImages: %v\n", changeStreamPreAndPostImages)
		}

		if oplogSize, ok := params["oplogSize"]; ok {
			fmt.Printf("✓ oplogSize: %v MB\n", oplogSize)
		}
	}

	// 7. 总结和建议
	fmt.Println("\n=== 诊断总结 ===")
	fmt.Println("如果 Change Stream 仍然不工作，请检查:")
	fmt.Println("1. 确保 MongoDB 运行在副本集模式")
	fmt.Println("2. 确保 MongoDB 版本 >= 3.6")
	fmt.Println("3. 确保有足够的权限访问 oplog")
	fmt.Println("4. 检查网络连接和防火墙设置")
	fmt.Println("5. 查看 MongoDB 日志文件中的错误信息")

	fmt.Println("\n如需初始化副本集，请执行:")
	fmt.Println("1. 停止 MongoDB")
	fmt.Println("2. 使用副本集模式启动: mongod --replSet rs0 --port 27017")
	fmt.Println("3. 连接到 MongoDB: mongo")
	fmt.Println("4. 初始化副本集: rs.initiate()")
	fmt.Println("5. 检查状态: rs.status()")
}

func main() {
	diagnoseMongoDB()
}
