package database

import (
	"encoding/json"
	"fmt"
	"gorm.io/driver/mysql"
	"gorm.io/gorm"
	"io/ioutil"
	"log"
	"os"
	"sync"
	"time"
)

type Node struct {
	IP   string `json:"ip"`
	Port int    `json:"port"`
}

type MetaConfig struct {
	Nodes []Node `json:"nodes"`
	Saas  int    `json:"saas"`
}

var DB *gorm.DB
var mu sync.Mutex

// loadConfig 读取并解析配置文件
func loadConfig(filePath string) (*MetaConfig, error) {
	file, err := os.Open(filePath)
	if err != nil {
		return nil, fmt.Errorf("无法打开配置文件: %v", err)
	}
	defer file.Close()

	byteValue, err := ioutil.ReadAll(file)
	if err != nil {
		return nil, fmt.Errorf("读取配置文件失败: %v", err)
	}

	var config MetaConfig
	if err := json.Unmarshal(byteValue, &config); err != nil {
		return nil, fmt.Errorf("解析配置文件失败: %v", err)
	}

	return &config, nil
}

// connectCluster 连接集群中的主节点
func connectCluster(config *MetaConfig) (*gorm.DB, error) {
	var primaryDB *gorm.DB

	for _, node := range config.Nodes {
		dsn := fmt.Sprintf("pgx:pgx_pwd@tcp(%s:%d)/kunlun_metadata_db?charset=utf8mb4&parseTime=True&loc=Local", node.IP, node.Port)
		conn, err := gorm.Open(mysql.Open(dsn), &gorm.Config{})
		if err != nil {
			log.Printf("无法连接到 %s:%d: %v", node.IP, node.Port, err)
			continue
		}

		// 检查该节点是否为主节点
		var isPrimary int
		if err := conn.Raw("SELECT @@global.read_only").Scan(&isPrimary).Error; err != nil {
			log.Printf("检查主节点失败: %v", err)
			continue
		}

		if isPrimary == 0 { // 主节点的 read_only 为 0
			primaryDB = conn
			break
		}
	}

	if primaryDB == nil {
		return nil, fmt.Errorf("未找到主节点")
	}

	return primaryDB, nil
}

// monitorPrimaryNode 定期监控主节点是否变化
func monitorPrimaryNode(config *MetaConfig) {
	for {
		time.Sleep(10 * time.Second) // 每30秒检查一次

		mu.Lock()
		var isPrimary int
		err := DB.Raw("SELECT @@global.read_only").Scan(&isPrimary).Error
		mu.Unlock()

		if err != nil || isPrimary != 0 {
			log.Println("主节点可能发生了变化，重新连接...")
			newDB, err := connectCluster(config)
			if err != nil {
				log.Printf("重新连接主节点失败: %v", err)
				continue
			}

			mu.Lock()
			DB = newDB
			mu.Unlock()
			log.Println("已连接到新的主节点")
		}
	}
}

// InitDB  初始化数据库连接
func InitDB(configPath string) {
	config, err := loadConfig(configPath)
	if err != nil {
		log.Fatalf("加载配置文件失败: %v", err)
	}

	DB, err = connectCluster(config)
	if err != nil {
		log.Fatalf("数据库连接失败: %v", err)
	}

	err = DB.AutoMigrate(&ClusterSteamTask{})
	if err != nil {
		log.Println("AutoMigrate->" + err.Error())
	}
	log.Println("已连接到主节点")

	go monitorPrimaryNode(config) // 启动主节点监控

}
